"""
LangChain Tools for ServiceNow Application Analysis
These tools are used by the LangGraph agent to query and analyze applications
"""

from typing import Any, Optional
from langchain_core.tools import tool
from servicenow_client import ServiceNowClient


# Initialize global client
_client: Optional[ServiceNowClient] = None


def get_client() -> ServiceNowClient:
    """Get or create ServiceNow client"""
    global _client
    if _client is None:
        _client = ServiceNowClient()
    return _client


def set_client(client: ServiceNowClient):
    """Set the ServiceNow client"""
    global _client
    _client = client


@tool
def get_portfolio_summary() -> dict:
    """
    Get a high-level summary of the entire application portfolio.
    Returns total count, breakdown by category, criticality, department, and cost metrics.
    Use this first to understand the overall landscape before drilling down.
    """
    client = get_client()
    return client.get_portfolio_summary()


@tool
def get_all_applications(limit: int = 100) -> list[dict]:
    """
    Retrieve all applications from ServiceNow CMDB.
    Returns application details including name, category, criticality, costs, and automation potential.
    Use limit parameter to control the number of results (default 100).
    """
    client = get_client()
    apps = client.get_all_applications(limit=limit)
    # Return simplified view for agent
    return [
        {
            "app_id": app.get("app_id"),
            "name": app.get("name"),
            "category": app.get("category"),
            "criticality": app.get("business_criticality"),
            "department": app.get("department"),
            "annual_cost": app.get("annual_cost_usd"),
            "user_count": app.get("user_count"),
            "automation_score": app.get("automation_potential", {}).get("overall_score", 0),
            "roi_percentage": app.get("roi_analysis", {}).get("roi_percentage", 0),
        }
        for app in apps
    ]


@tool
def get_applications_by_category(category: str) -> list[dict]:
    """
    Get applications filtered by a specific category.
    Valid categories include: Trading & Markets, Risk & Compliance, Core Banking,
    Customer Relationship, Data & Analytics, Infrastructure & Security, 
    Human Capital, Operations & Workflow, Integration & API, Finance & Accounting
    """
    client = get_client()
    apps = client.get_applications_by_category(category)
    return [
        {
            "app_id": app.get("app_id"),
            "name": app.get("name"),
            "criticality": app.get("business_criticality"),
            "automation_score": app.get("automation_potential", {}).get("overall_score", 0),
            "roi_percentage": app.get("roi_analysis", {}).get("roi_percentage", 0),
            "annual_savings": app.get("roi_analysis", {}).get("annual_savings_usd", 0),
        }
        for app in apps
    ]


@tool
def get_applications_by_criticality(criticality: str) -> list[dict]:
    """
    Get applications filtered by business criticality level.
    Valid levels: Critical, High, Medium, Low
    """
    client = get_client()
    apps = client.get_applications_by_criticality(criticality)
    return [
        {
            "app_id": app.get("app_id"),
            "name": app.get("name"),
            "category": app.get("category"),
            "automation_score": app.get("automation_potential", {}).get("overall_score", 0),
            "roi_percentage": app.get("roi_analysis", {}).get("roi_percentage", 0),
        }
        for app in apps
    ]


@tool
def get_application_details(app_id: str) -> dict:
    """
    Get detailed information about a specific application including full description,
    technology stack, automation opportunities, and ROI analysis.
    Use this to drill down into promising automation candidates.
    """
    client = get_client()
    app = client.get_application_by_id(app_id)
    if not app:
        return {"error": f"Application {app_id} not found"}
    return app


@tool
def get_high_automation_candidates(min_score: float = 0.5, limit: int = 50) -> list[dict]:
    """
    Get applications with high automation potential.
    min_score: Minimum automation score (0.0 to 1.0, default 0.5)
    limit: Maximum number of results (default 50)
    Returns applications sorted by automation potential, best candidates first.
    """
    client = get_client()
    candidates = client.get_high_automation_candidates(min_score)[:limit]
    return [
        {
            "app_id": app.get("app_id"),
            "name": app.get("name"),
            "category": app.get("category"),
            "criticality": app.get("business_criticality"),
            "automation_score": app.get("automation_potential", {}).get("overall_score", 0),
            "automation_opportunities": list(app.get("automation_potential", {}).get("opportunities", {}).keys()),
            "roi_percentage": app.get("roi_analysis", {}).get("roi_percentage", 0),
            "annual_savings": app.get("roi_analysis", {}).get("annual_savings_usd", 0),
            "payback_months": app.get("roi_analysis", {}).get("payback_months", 0),
        }
        for app in candidates
    ]


@tool
def get_top_roi_opportunities(limit: int = 50) -> list[dict]:
    """
    Get applications with the best ROI potential for automation.
    Returns applications sorted by ROI percentage, highest first.
    Includes annual savings, implementation cost, and payback period.
    """
    client = get_client()
    opportunities = client.get_top_roi_opportunities(limit)
    return [
        {
            "app_id": app.get("app_id"),
            "name": app.get("name"),
            "category": app.get("category"),
            "criticality": app.get("business_criticality"),
            "roi_percentage": app.get("roi_analysis", {}).get("roi_percentage", 0),
            "annual_savings": app.get("roi_analysis", {}).get("annual_savings_usd", 0),
            "implementation_cost": app.get("roi_analysis", {}).get("implementation_cost_usd", 0),
            "payback_months": app.get("roi_analysis", {}).get("payback_months", 0),
            "npv_3yr": app.get("roi_analysis", {}).get("net_present_value_3yr", 0),
            "automation_score": app.get("automation_potential", {}).get("overall_score", 0),
        }
        for app in opportunities
    ]


@tool  
def search_applications(query: str) -> list[dict]:
    """
    Search applications by name, description, or category.
    Use this to find specific applications or application types.
    """
    client = get_client()
    results = client.search_applications(query)
    return [
        {
            "app_id": app.get("app_id"),
            "name": app.get("name"),
            "category": app.get("category"),
            "description": app.get("description", "")[:200] + "...",
            "automation_score": app.get("automation_potential", {}).get("overall_score", 0),
            "roi_percentage": app.get("roi_analysis", {}).get("roi_percentage", 0),
        }
        for app in results[:20]  # Limit search results
    ]


@tool
def get_automation_analysis_by_department(department: str) -> dict:
    """
    Get automation analysis summary for a specific department.
    Returns aggregate metrics including total apps, potential savings, and top candidates.
    """
    client = get_client()
    apps = client.get_applications_by_department(department)
    
    if not apps:
        return {"error": f"No applications found for department: {department}"}
    
    total_savings = sum(
        app.get("roi_analysis", {}).get("annual_savings_usd", 0) 
        for app in apps
    )
    total_cost = sum(app.get("annual_cost_usd", 0) for app in apps)
    avg_automation_score = sum(
        app.get("automation_potential", {}).get("overall_score", 0) 
        for app in apps
    ) / len(apps)
    
    # Top candidates in this department
    sorted_apps = sorted(
        apps,
        key=lambda x: x.get("roi_analysis", {}).get("roi_percentage", 0),
        reverse=True
    )
    
    return {
        "department": department,
        "total_applications": len(apps),
        "total_annual_cost": total_cost,
        "total_potential_savings": total_savings,
        "average_automation_score": round(avg_automation_score, 2),
        "top_candidates": [
            {
                "name": app.get("name"),
                "roi_percentage": app.get("roi_analysis", {}).get("roi_percentage", 0),
                "annual_savings": app.get("roi_analysis", {}).get("annual_savings_usd", 0),
            }
            for app in sorted_apps[:5]
        ]
    }


@tool
def calculate_portfolio_automation_summary() -> dict:
    """
    Calculate comprehensive automation summary for the entire portfolio.
    Returns total potential savings, top opportunities by category,
    and recommended prioritization tiers.
    """
    client = get_client()
    all_apps = client.get_all_applications()
    
    # Calculate totals
    total_potential_savings = sum(
        app.get("roi_analysis", {}).get("annual_savings_usd", 0)
        for app in all_apps
    )
    total_implementation_cost = sum(
        app.get("roi_analysis", {}).get("implementation_cost_usd", 0)
        for app in all_apps
    )
    
    # Group by category
    by_category = {}
    for app in all_apps:
        cat = app.get("category", "Unknown")
        if cat not in by_category:
            by_category[cat] = {"count": 0, "savings": 0, "avg_roi": 0, "roi_sum": 0}
        by_category[cat]["count"] += 1
        by_category[cat]["savings"] += app.get("roi_analysis", {}).get("annual_savings_usd", 0)
        by_category[cat]["roi_sum"] += app.get("roi_analysis", {}).get("roi_percentage", 0)
    
    for cat in by_category:
        by_category[cat]["avg_roi"] = round(
            by_category[cat]["roi_sum"] / by_category[cat]["count"], 1
        )
        del by_category[cat]["roi_sum"]
    
    # Priority tiers
    high_priority = [
        app for app in all_apps
        if app.get("automation_potential", {}).get("overall_score", 0) >= 0.7
        and app.get("roi_analysis", {}).get("roi_percentage", 0) >= 50
    ]
    medium_priority = [
        app for app in all_apps
        if 0.5 <= app.get("automation_potential", {}).get("overall_score", 0) < 0.7
        and 20 <= app.get("roi_analysis", {}).get("roi_percentage", 0) < 50
    ]
    
    return {
        "total_applications_analyzed": len(all_apps),
        "total_annual_savings_potential": total_potential_savings,
        "total_implementation_cost": total_implementation_cost,
        "overall_portfolio_roi": round(
            (total_potential_savings / max(total_implementation_cost, 1)) * 100, 1
        ),
        "by_category": by_category,
        "priority_tiers": {
            "high": {
                "count": len(high_priority),
                "total_savings": sum(
                    app.get("roi_analysis", {}).get("annual_savings_usd", 0)
                    for app in high_priority
                )
            },
            "medium": {
                "count": len(medium_priority),
                "total_savings": sum(
                    app.get("roi_analysis", {}).get("annual_savings_usd", 0)
                    for app in medium_priority
                )
            }
        }
    }


# List of all tools for the agent
ALL_TOOLS = [
    get_portfolio_summary,
    get_all_applications,
    get_applications_by_category,
    get_applications_by_criticality,
    get_application_details,
    get_high_automation_candidates,
    get_top_roi_opportunities,
    search_applications,
    get_automation_analysis_by_department,
    calculate_portfolio_automation_summary,
]

