"""
ServiceNow API Client
Reads business applications from ServiceNow cmdb_ci_business_app table
"""

import json
from typing import Any, Optional
from datetime import datetime
import requests
from requests.auth import HTTPBasicAuth

from config import get_settings


class ServiceNowClient:
    """Client for interacting with ServiceNow CMDB Business Applications"""
    
    def __init__(self, instance_url: str = None, username: str = None, password: str = None):
        settings = get_settings()
        self.instance_url = (instance_url or settings.servicenow_instance_url).rstrip('/')
        self.username = username or settings.servicenow_username
        self.password = password or settings.servicenow_password
        self.use_mock_data = settings.use_mock_data
        self.mock_data_path = settings.mock_data_path
        self._cache = None
        
        # Business Applications table
        self.table = "cmdb_ci_business_app"
        
    def _get_auth(self) -> HTTPBasicAuth:
        """Get HTTP Basic Auth object"""
        return HTTPBasicAuth(self.username, self.password)
    
    def _get_headers(self) -> dict:
        """Get default headers for API requests"""
        return {
            "Content-Type": "application/json",
            "Accept": "application/json"
        }
    
    def _parse_cost(self, cost_str: str) -> int:
        """Parse cost string to integer, handling commas and decimals"""
        if not cost_str:
            return 0
        try:
            # Remove commas and convert
            return int(float(str(cost_str).replace(",", "").replace("$", "")))
        except:
            return 0
    
    def _parse_app_data(self, raw_app: dict) -> dict:
        """Parse ServiceNow app record into our format"""
        # Extract extended data from comments field (JSON)
        comments = raw_app.get("comments", "")
        extended_data = {}
        try:
            if comments and comments.strip().startswith("{"):
                extended_data = json.loads(comments)
        except:
            pass
        
        # Map criticality back
        crit_map = {
            "1 - most critical": "Critical",
            "2 - somewhat critical": "High",
            "3 - less critical": "Medium", 
            "4 - not critical": "Low"
        }
        
        return {
            "sys_id": raw_app.get("sys_id", ""),
            "app_id": extended_data.get("app_id", raw_app.get("number", "")),
            "name": raw_app.get("name", "Unknown"),
            "description": raw_app.get("short_description", ""),
            "category": extended_data.get("category", raw_app.get("used_for", "")),
            "subcategory": extended_data.get("subcategory", ""),
            "department": extended_data.get("department", ""),
            "vendor": extended_data.get("vendor", raw_app.get("manufacturer", "")),
            "business_criticality": crit_map.get(
                raw_app.get("busines_criticality", ""), 
                "Medium"
            ),
            "lifecycle_stage": "Production" if raw_app.get("operational_status") == "1" else "Other",
            "technology_stack": extended_data.get("technology_stack", {}),
            "automation_potential": extended_data.get("automation_potential", {"overall_score": 0.5}),
            "roi_analysis": extended_data.get("roi_analysis", {}),
            "annual_cost_usd": extended_data.get("annual_cost_usd", self._parse_cost(raw_app.get("cost", "0"))),
            "user_count": extended_data.get("user_count", 0),
            "transaction_volume_daily": extended_data.get("transaction_volume_daily", 0),
            "data_classification": extended_data.get("data_classification", ""),
            "regulatory_scope": extended_data.get("regulatory_scope", []),
        }
    
    def _fetch_from_servicenow(self, limit: int = None, offset: int = 0) -> list[dict]:
        """Fetch applications from ServiceNow API"""
        params = {
            "sysparm_offset": offset,
            "sysparm_display_value": "true"
        }
        if limit:
            params["sysparm_limit"] = limit
        else:
            params["sysparm_limit"] = 2000  # Get all
            
        try:
            response = requests.get(
                f"{self.instance_url}/api/now/table/{self.table}",
                auth=self._get_auth(),
                headers=self._get_headers(),
                params=params,
                timeout=60
            )
            response.raise_for_status()
            raw_apps = response.json().get("result", [])
            
            # Parse each app
            return [self._parse_app_data(app) for app in raw_apps]
            
        except requests.exceptions.RequestException as e:
            print(f"[ERROR] Failed to fetch from ServiceNow: {str(e)}")
            raise ServiceNowAPIError(f"API request failed: {str(e)}")
    
    def get_all_applications(self, limit: int = None, offset: int = 0) -> list[dict]:
        """
        Retrieve all business applications from ServiceNow
        """
        if self.use_mock_data:
            return self._get_mock_applications(limit, offset)
        
        # Fetch from real ServiceNow
        if self._cache is None:
            print("[*] Fetching applications from ServiceNow...")
            self._cache = self._fetch_from_servicenow()
            print(f"[OK] Retrieved {len(self._cache)} applications")
        
        apps = self._cache
        if limit:
            return apps[offset:offset + limit]
        return apps[offset:]
    
    def _get_mock_applications(self, limit: int = None, offset: int = 0) -> list[dict]:
        """Load from mock data file"""
        import os
        paths = [
            self.mock_data_path,
            os.path.join(os.path.dirname(__file__), self.mock_data_path),
            os.path.join(os.path.dirname(__file__), "data", "servicenow_apps.json"),
        ]
        
        for path in paths:
            if os.path.exists(path):
                with open(path, 'r', encoding='utf-8') as f:
                    data = json.load(f)
                    apps = data.get("applications", [])
                    if limit:
                        return apps[offset:offset + limit]
                    return apps[offset:]
        
        raise FileNotFoundError("Mock data file not found")
    
    def get_application_by_id(self, app_id: str) -> Optional[dict]:
        """Retrieve a specific application by ID"""
        apps = self.get_all_applications()
        for app in apps:
            if app.get("app_id") == app_id or app.get("sys_id") == app_id:
                return app
        return None
    
    def get_applications_by_category(self, category: str) -> list[dict]:
        """Get applications filtered by category"""
        apps = self.get_all_applications()
        return [
            app for app in apps
            if app.get("category", "").lower() == category.lower()
        ]
    
    def get_applications_by_criticality(self, criticality: str) -> list[dict]:
        """Get applications filtered by business criticality"""
        apps = self.get_all_applications()
        return [
            app for app in apps
            if app.get("business_criticality", "").lower() == criticality.lower()
        ]
    
    def get_applications_by_department(self, department: str) -> list[dict]:
        """Get applications filtered by department"""
        apps = self.get_all_applications()
        return [
            app for app in apps
            if department.lower() in app.get("department", "").lower()
        ]
    
    def get_portfolio_summary(self) -> dict:
        """Get summary statistics of the application portfolio"""
        apps = self.get_all_applications()
        
        categories = {}
        criticality_counts = {}
        departments = {}
        total_cost = 0
        total_users = 0
        
        for app in apps:
            cat = app.get("category", "Unknown")
            categories[cat] = categories.get(cat, 0) + 1
            
            crit = app.get("business_criticality", "Unknown")
            criticality_counts[crit] = criticality_counts.get(crit, 0) + 1
            
            dept = app.get("department", "Unknown")
            departments[dept] = departments.get(dept, 0) + 1
            
            total_cost += app.get("annual_cost_usd", 0)
            total_users += app.get("user_count", 0)
        
        return {
            "total_applications": len(apps),
            "by_category": categories,
            "by_criticality": criticality_counts,
            "by_department": departments,
            "total_annual_cost_usd": total_cost,
            "total_user_count": total_users,
            "data_source": "mock" if self.use_mock_data else "servicenow"
        }
    
    def get_high_automation_candidates(self, min_score: float = 0.5) -> list[dict]:
        """Get applications with high automation potential"""
        apps = self.get_all_applications()
        candidates = [
            app for app in apps
            if app.get("automation_potential", {}).get("overall_score", 0) >= min_score
        ]
        candidates.sort(
            key=lambda x: x.get("automation_potential", {}).get("overall_score", 0),
            reverse=True
        )
        return candidates
    
    def get_top_roi_opportunities(self, limit: int = 50) -> list[dict]:
        """Get applications with best ROI potential"""
        apps = self.get_all_applications()
        positive_roi = [
            app for app in apps
            if app.get("roi_analysis", {}).get("roi_percentage", 0) > 0
        ]
        positive_roi.sort(
            key=lambda x: x.get("roi_analysis", {}).get("roi_percentage", 0),
            reverse=True
        )
        return positive_roi[:limit]
    
    def search_applications(self, query: str) -> list[dict]:
        """Search applications by name or description"""
        apps = self.get_all_applications()
        query_lower = query.lower()
        return [
            app for app in apps
            if (query_lower in app.get("name", "").lower() or
                query_lower in app.get("description", "").lower() or
                query_lower in app.get("category", "").lower())
        ][:20]


class ServiceNowAPIError(Exception):
    """Custom exception for ServiceNow API errors"""
    pass


def create_client(**kwargs) -> ServiceNowClient:
    """Create a ServiceNow client"""
    return ServiceNowClient(**kwargs)


if __name__ == "__main__":
    # Test the client reading from ServiceNow
    print("=" * 60)
    print("Testing ServiceNow Client")
    print("=" * 60)
    
    client = ServiceNowClient()
    
    print("\n[*] Portfolio Summary:")
    summary = client.get_portfolio_summary()
    print(f"    Data Source: {summary['data_source']}")
    print(f"    Total Applications: {summary['total_applications']}")
    print(f"    Total Annual Cost: ${summary['total_annual_cost_usd']:,.0f}")
    
    print("\n[*] Top 5 ROI Opportunities:")
    top_roi = client.get_top_roi_opportunities(5)
    for app in top_roi:
        roi = app.get("roi_analysis", {})
        print(f"    - {app['name'][:40]}: ROI {roi.get('roi_percentage', 0):.0f}%")
