"""
Full AI-Powered Analysis with Enhanced Excel Report
Reads all credentials from .env file
"""
import os
from datetime import datetime

from config import get_settings

# Load settings from .env
settings = get_settings()

# Set OpenAI API key from config
if settings.openai_api_key:
    os.environ["OPENAI_API_KEY"] = settings.openai_api_key

from rich.console import Console
from rich.markdown import Markdown
from rich.panel import Panel
from langchain_core.messages import AIMessage, HumanMessage

from agent import ApplicationAnalyzerAgent
from servicenow_client import ServiceNowClient
from enhanced_report_generator import EnhancedReportGenerator

console = Console()


def run_full_analysis():
    console.print("\n" + "=" * 70)
    console.print("[bold cyan]ServiceNow Gen AI Automation Analyzer[/bold cyan]")
    console.print("[dim]Full Analysis with AI-Enhanced Excel Report[/dim]")
    console.print("=" * 70 + "\n")
    
    # Check configuration
    if not settings.servicenow_password:
        console.print("[red]ERROR: SERVICENOW_PASSWORD not set in .env file![/red]")
        console.print("[yellow]Copy env_template.txt to .env and fill in your credentials.[/yellow]")
        return
    
    if not settings.openai_api_key:
        console.print("[red]ERROR: OPENAI_API_KEY not set in .env file![/red]")
        console.print("[yellow]Copy env_template.txt to .env and fill in your credentials.[/yellow]")
        return
    
    console.print(f"[dim]ServiceNow: {settings.servicenow_instance_url}[/dim]")
    console.print(f"[dim]OpenAI Model: {settings.openai_model}[/dim]\n")
    
    # Step 1: Connect to ServiceNow
    console.print("[yellow][1/4] Connecting to ServiceNow...[/yellow]")
    client = ServiceNowClient()
    apps = client.get_all_applications()
    console.print(f"[green]      Retrieved {len(apps)} applications[/green]\n")
    
    # Step 2: Initialize AI agent
    console.print("[yellow][2/4] Initializing AI Agent (GPT-4)...[/yellow]")
    agent = ApplicationAnalyzerAgent(client)
    
    if agent.llm is None:
        console.print("[red]ERROR: Failed to initialize OpenAI![/red]")
        return
    console.print("[green]      AI Agent ready[/green]\n")
    
    # Step 3: Run AI analysis
    console.print("[yellow][3/4] Running AI Analysis...[/yellow]")
    console.print("[dim]      GPT-4 is analyzing your portfolio...[/dim]\n")
    
    query = """Analyze our ServiceNow application portfolio for a financial services company. 

Please:
1. Get the portfolio summary
2. Identify top automation candidates (score >= 0.7)
3. Find best ROI opportunities
4. Analyze by category

Provide executive recommendations including:
- Top 10 apps to automate with Gen AI (with specific metrics)
- Expected ROI and savings
- Implementation priority tiers
- Key risks and mitigation strategies

Be specific and data-driven for CIO/CFO audience."""

    result = agent.run(query)
    
    # Extract AI response
    ai_response = ""
    for msg in result.get("messages", []):
        if isinstance(msg, AIMessage):
            ai_response = msg.content
            break
    
    console.print("[green]      AI Analysis complete[/green]\n")
    
    # Step 4: Generate enhanced Excel report
    console.print("[yellow][4/4] Generating Enhanced Excel Report...[/yellow]")
    
    generator = EnhancedReportGenerator(client)
    report_path = generator.generate_report(ai_insights=ai_response)
    
    console.print(f"[green]      Report saved![/green]\n")
    
    # Display AI insights
    console.print("=" * 70)
    console.print("[bold green]AI ANALYSIS RESULTS[/bold green]")
    console.print("=" * 70 + "\n")
    
    if ai_response:
        console.print(Markdown(ai_response))
    
    # Summary
    console.print("\n" + "=" * 70)
    console.print("[bold cyan]ANALYSIS COMPLETE[/bold cyan]")
    console.print("=" * 70)
    console.print(f"\n[bold]Excel Report:[/bold] {report_path}")
    console.print("\nThe report includes:")
    console.print("  - AI Insights (GPT-4 recommendations)")
    console.print("  - Executive Summary")
    console.print("  - Top 50 Recommendations")  
    console.print("  - Automation Candidates")
    console.print("  - Category & Department Analysis")
    console.print("  - Implementation Roadmap")
    console.print("  - Full Portfolio Data")
    console.print("=" * 70 + "\n")
    
    return report_path


if __name__ == "__main__":
    run_full_analysis()
