"""
Run AI-Powered Analysis using OpenAI GPT-4
Reads all credentials from .env file
"""
import os

from config import get_settings

# Load settings from .env
settings = get_settings()

# Set OpenAI API key from config
if settings.openai_api_key:
    os.environ["OPENAI_API_KEY"] = settings.openai_api_key

from rich.console import Console
from rich.markdown import Markdown
from rich.panel import Panel
from langchain_core.messages import AIMessage, HumanMessage

from agent import ApplicationAnalyzerAgent
from servicenow_client import ServiceNowClient

console = Console()


def run_analysis():
    console.print("\n" + "=" * 70)
    console.print("[bold cyan]ServiceNow Gen AI Automation Analyzer[/bold cyan]")
    console.print("[dim]Powered by OpenAI GPT-4 + LangGraph[/dim]")
    console.print("=" * 70 + "\n")
    
    # Check configuration
    if not settings.openai_api_key:
        console.print("[red]ERROR: OPENAI_API_KEY not set in .env file![/red]")
        console.print("[yellow]Copy env_template.txt to .env and fill in your credentials.[/yellow]")
        return
    
    # Initialize client to fetch from ServiceNow
    console.print("[yellow][1/3] Connecting to ServiceNow...[/yellow]")
    client = ServiceNowClient()
    summary = client.get_portfolio_summary()
    console.print(f"[green]      Retrieved {summary['total_applications']} applications from ServiceNow[/green]\n")
    
    # Initialize AI agent
    console.print("[yellow][2/3] Initializing AI Agent (GPT-4)...[/yellow]")
    agent = ApplicationAnalyzerAgent(client)
    
    if agent.llm is None:
        console.print("[red]ERROR: OpenAI API key not configured![/red]")
        return
    
    console.print("[green]      AI Agent ready with tools bound[/green]\n")
    
    # Run analysis
    console.print("[yellow][3/3] Running AI Analysis...[/yellow]")
    console.print("[dim]      The AI will analyze your portfolio and provide recommendations...[/dim]\n")
    
    query = """Analyze our ServiceNow application portfolio for a financial services company. 

I need you to:
1. Get the portfolio summary to understand the overall landscape
2. Identify the top applications with highest automation potential
3. Find the best ROI opportunities
4. Analyze by category to see which areas have most potential

Then provide executive-level recommendations for CIO/CFO including:
- Top 10 applications to automate with Gen AI
- Expected ROI and annual savings
- Implementation priority (Quick Wins vs Strategic)
- Key risks to consider

Be specific with numbers and data-driven insights."""

    result = agent.run(query)
    
    # Display results
    console.print("\n" + "=" * 70)
    console.print("[bold green]AI ANALYSIS COMPLETE[/bold green]")
    console.print("=" * 70 + "\n")
    
    for msg in result.get("messages", []):
        if isinstance(msg, AIMessage):
            console.print(Markdown(msg.content))
    
    if result.get("error"):
        console.print(f"\n[red]Error: {result['error']}[/red]")
    
    console.print("\n" + "=" * 70)
    console.print("[bold cyan]Analysis complete![/bold cyan]")
    console.print("=" * 70)


if __name__ == "__main__":
    run_analysis()
