"""
Excel Report Generator for Application Portfolio Analysis
Generates comprehensive executive reports with ROI analysis and recommendations
"""

import os
from datetime import datetime
from typing import Any
import pandas as pd
from openpyxl import Workbook
from openpyxl.styles import (
    Font, PatternFill, Alignment, Border, Side,
    NamedStyle, GradientFill
)
from openpyxl.chart import BarChart, PieChart, Reference, LineChart
from openpyxl.chart.label import DataLabelList
from openpyxl.utils.dataframe import dataframe_to_rows
from openpyxl.formatting.rule import ColorScaleRule, FormulaRule
from openpyxl.utils import get_column_letter

from config import get_settings
from servicenow_client import ServiceNowClient


class ExcelReportGenerator:
    """Generates comprehensive Excel reports for CIO/CFO decision-making"""
    
    def __init__(self, client: ServiceNowClient = None):
        self.client = client or ServiceNowClient()
        self.settings = get_settings()
        
        # Style definitions
        self.header_font = Font(bold=True, color="FFFFFF", size=11)
        self.header_fill = PatternFill(start_color="1F4E79", end_color="1F4E79", fill_type="solid")
        self.subheader_fill = PatternFill(start_color="2E75B6", end_color="2E75B6", fill_type="solid")
        self.highlight_fill = PatternFill(start_color="E2EFDA", end_color="E2EFDA", fill_type="solid")
        self.warning_fill = PatternFill(start_color="FCE4D6", end_color="FCE4D6", fill_type="solid")
        self.border = Border(
            left=Side(style='thin'),
            right=Side(style='thin'),
            top=Side(style='thin'),
            bottom=Side(style='thin')
        )
        self.money_format = '_("$"* #,##0_);_("$"* (#,##0);_("$"* "-"_);_(@_)'
        self.percent_format = '0.0%'
    
    def generate_report(self, output_path: str = None) -> str:
        """Generate the complete Excel report"""
        if output_path is None:
            os.makedirs(self.settings.output_directory, exist_ok=True)
            timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
            output_path = os.path.join(
                self.settings.output_directory, 
                f"GenAI_Automation_ROI_Analysis_{timestamp}.xlsx"
            )
        
        # Fetch all data
        print("[*] Fetching application data...")
        all_apps = self.client.get_all_applications()
        summary = self.client.get_portfolio_summary()
        top_roi = self.client.get_top_roi_opportunities(100)
        high_automation = self.client.get_high_automation_candidates(0.5)
        
        # Create workbook
        wb = Workbook()
        
        # Create sheets
        self._create_executive_summary(wb, summary, all_apps)
        self._create_top_recommendations(wb, top_roi)
        self._create_automation_candidates(wb, high_automation)
        self._create_category_analysis(wb, all_apps)
        self._create_department_analysis(wb, all_apps)
        self._create_implementation_roadmap(wb, top_roi, high_automation)
        self._create_full_portfolio(wb, all_apps)
        self._create_roi_details(wb, all_apps)
        
        # Remove default sheet
        if "Sheet" in wb.sheetnames:
            del wb["Sheet"]
        
        # Save workbook
        wb.save(output_path)
        print(f"[OK] Report saved to: {output_path}")
        
        return output_path
    
    def _create_executive_summary(self, wb: Workbook, summary: dict, all_apps: list):
        """Create executive summary sheet"""
        ws = wb.create_sheet("Executive Summary", 0)
        
        # Title
        ws.merge_cells('A1:H1')
        ws['A1'] = "Gen AI Automation ROI Analysis - Executive Summary"
        ws['A1'].font = Font(bold=True, size=18, color="1F4E79")
        ws['A1'].alignment = Alignment(horizontal='center')
        
        # Subtitle
        ws.merge_cells('A2:H2')
        ws['A2'] = f"Generated: {datetime.now().strftime('%B %d, %Y at %I:%M %p')}"
        ws['A2'].font = Font(italic=True, size=10, color="666666")
        ws['A2'].alignment = Alignment(horizontal='center')
        
        # Key Metrics Section
        row = 4
        ws.merge_cells(f'A{row}:D{row}')
        ws[f'A{row}'] = "Portfolio Overview"
        ws[f'A{row}'].font = Font(bold=True, size=14)
        ws[f'A{row}'].fill = self.header_fill
        ws[f'A{row}'].font = self.header_font
        
        row += 1
        metrics = [
            ("Total Applications", summary['total_applications'], ""),
            ("Total Annual IT Spend", summary['total_annual_cost_usd'], "$"),
            ("Total Users Supported", summary['total_user_count'], ""),
        ]
        
        for label, value, prefix in metrics:
            ws[f'A{row}'] = label
            ws[f'A{row}'].font = Font(bold=True)
            if prefix == "$":
                ws[f'B{row}'] = value
                ws[f'B{row}'].number_format = self.money_format
            else:
                ws[f'B{row}'] = f"{value:,}" if isinstance(value, (int, float)) else value
            row += 1
        
        # Calculate totals for automation
        total_savings = sum(app.get("roi_analysis", {}).get("annual_savings_usd", 0) for app in all_apps)
        total_impl_cost = sum(app.get("roi_analysis", {}).get("implementation_cost_usd", 0) for app in all_apps)
        avg_roi = sum(app.get("roi_analysis", {}).get("roi_percentage", 0) for app in all_apps) / len(all_apps)
        
        row += 1
        ws.merge_cells(f'A{row}:D{row}')
        ws[f'A{row}'] = "Automation Opportunity Summary"
        ws[f'A{row}'].font = self.header_font
        ws[f'A{row}'].fill = self.header_fill
        
        row += 1
        automation_metrics = [
            ("Total Potential Annual Savings", total_savings, "$"),
            ("Total Implementation Investment", total_impl_cost, "$"),
            ("Average ROI Across Portfolio", avg_roi, "%"),
            ("Apps with ROI > 50%", len([a for a in all_apps if a.get("roi_analysis", {}).get("roi_percentage", 0) > 50]), ""),
            ("Apps with Automation Score > 0.7", len([a for a in all_apps if a.get("automation_potential", {}).get("overall_score", 0) > 0.7]), ""),
        ]
        
        for label, value, prefix in automation_metrics:
            ws[f'A{row}'] = label
            ws[f'A{row}'].font = Font(bold=True)
            if prefix == "$":
                ws[f'B{row}'] = value
                ws[f'B{row}'].number_format = self.money_format
            elif prefix == "%":
                ws[f'B{row}'] = f"{value:.1f}%"
            else:
                ws[f'B{row}'] = f"{value:,}" if isinstance(value, (int, float)) else value
            ws[f'A{row}'].fill = self.highlight_fill
            ws[f'B{row}'].fill = self.highlight_fill
            row += 1
        
        # Top 5 Quick Wins
        row += 2
        ws.merge_cells(f'A{row}:F{row}')
        ws[f'A{row}'] = "Top 5 Quick Win Recommendations"
        ws[f'A{row}'].font = self.header_font
        ws[f'A{row}'].fill = PatternFill(start_color="70AD47", end_color="70AD47", fill_type="solid")
        
        row += 1
        headers = ["Application", "Category", "ROI %", "Annual Savings", "Payback (Months)", "Automation Score"]
        for col, header in enumerate(headers, 1):
            cell = ws.cell(row=row, column=col, value=header)
            cell.font = Font(bold=True)
            cell.fill = self.subheader_fill
            cell.font = Font(bold=True, color="FFFFFF")
            cell.border = self.border
        
        # Sort by ROI and payback for quick wins
        quick_wins = sorted(
            [a for a in all_apps if a.get("roi_analysis", {}).get("payback_months", 100) < 12],
            key=lambda x: (-x.get("roi_analysis", {}).get("roi_percentage", 0), 
                          x.get("roi_analysis", {}).get("payback_months", 100))
        )[:5]
        
        row += 1
        for app in quick_wins:
            roi = app.get("roi_analysis", {})
            auto = app.get("automation_potential", {})
            ws.cell(row=row, column=1, value=app.get("name", "")[:40]).border = self.border
            ws.cell(row=row, column=2, value=app.get("category", "")).border = self.border
            ws.cell(row=row, column=3, value=roi.get("roi_percentage", 0) / 100).number_format = self.percent_format
            ws.cell(row=row, column=3).border = self.border
            ws.cell(row=row, column=4, value=roi.get("annual_savings_usd", 0)).number_format = self.money_format
            ws.cell(row=row, column=4).border = self.border
            ws.cell(row=row, column=5, value=roi.get("payback_months", 0)).border = self.border
            ws.cell(row=row, column=6, value=auto.get("overall_score", 0)).number_format = '0.00'
            ws.cell(row=row, column=6).border = self.border
            row += 1
        
        # Adjust column widths
        ws.column_dimensions['A'].width = 40
        ws.column_dimensions['B'].width = 25
        ws.column_dimensions['C'].width = 12
        ws.column_dimensions['D'].width = 18
        ws.column_dimensions['E'].width = 18
        ws.column_dimensions['F'].width = 18
    
    def _create_top_recommendations(self, wb: Workbook, top_roi: list):
        """Create top recommendations sheet"""
        ws = wb.create_sheet("Top 50 Recommendations")
        
        # Title
        ws.merge_cells('A1:J1')
        ws['A1'] = "Top 50 Applications by ROI Potential"
        ws['A1'].font = Font(bold=True, size=16, color="1F4E79")
        
        # Headers
        headers = [
            "Rank", "Application Name", "Category", "Criticality", 
            "ROI %", "Annual Savings", "Implementation Cost", 
            "Payback (Months)", "3-Year NPV", "Automation Score"
        ]
        
        for col, header in enumerate(headers, 1):
            cell = ws.cell(row=3, column=col, value=header)
            cell.font = self.header_font
            cell.fill = self.header_fill
            cell.border = self.border
            cell.alignment = Alignment(horizontal='center', wrap_text=True)
        
        # Data
        for idx, app in enumerate(top_roi[:50], 1):
            row = idx + 3
            roi = app.get("roi_analysis", {})
            auto = app.get("automation_potential", {})
            
            data = [
                idx,
                app.get("name", "")[:45],
                app.get("category", ""),
                app.get("business_criticality", ""),
                roi.get("roi_percentage", 0) / 100,
                roi.get("annual_savings_usd", 0),
                roi.get("implementation_cost_usd", 0),
                roi.get("payback_months", 0),
                roi.get("net_present_value_3yr", 0),
                auto.get("overall_score", 0)
            ]
            
            for col, value in enumerate(data, 1):
                cell = ws.cell(row=row, column=col, value=value)
                cell.border = self.border
                
                if col == 5:  # ROI %
                    cell.number_format = self.percent_format
                elif col in [6, 7, 9]:  # Money columns
                    cell.number_format = self.money_format
                elif col == 10:  # Automation score
                    cell.number_format = '0.00'
            
            # Highlight high ROI rows
            if roi.get("roi_percentage", 0) > 75:
                for col in range(1, 11):
                    ws.cell(row=row, column=col).fill = self.highlight_fill
        
        # Add color scale for ROI column
        ws.conditional_formatting.add(
            'E4:E53',
            ColorScaleRule(
                start_type='min', start_color='F8696B',
                mid_type='percentile', mid_value=50, mid_color='FFEB84',
                end_type='max', end_color='63BE7B'
            )
        )
        
        # Adjust column widths
        widths = [6, 45, 22, 12, 10, 16, 18, 14, 16, 14]
        for i, width in enumerate(widths, 1):
            ws.column_dimensions[get_column_letter(i)].width = width
        
        # Freeze header row
        ws.freeze_panes = 'A4'
    
    def _create_automation_candidates(self, wb: Workbook, candidates: list):
        """Create automation candidates analysis sheet"""
        ws = wb.create_sheet("Automation Candidates")
        
        # Title
        ws.merge_cells('A1:I1')
        ws['A1'] = "High Automation Potential Applications"
        ws['A1'].font = Font(bold=True, size=16, color="1F4E79")
        
        ws['A2'] = "Applications with automation score ≥ 0.5"
        ws['A2'].font = Font(italic=True, color="666666")
        
        # Headers
        headers = [
            "Application", "Category", "Automation Score", "Top Opportunities",
            "ROI %", "Annual Savings", "Payback", "Criticality", "Department"
        ]
        
        for col, header in enumerate(headers, 1):
            cell = ws.cell(row=4, column=col, value=header)
            cell.font = self.header_font
            cell.fill = self.header_fill
            cell.border = self.border
        
        # Data
        for idx, app in enumerate(candidates[:100], 1):
            row = idx + 4
            roi = app.get("roi_analysis", {})
            auto = app.get("automation_potential", {})
            opportunities = auto.get("opportunities", {})
            top_opps = ", ".join(list(opportunities.keys())[:3])
            
            data = [
                app.get("name", "")[:40],
                app.get("category", ""),
                auto.get("overall_score", 0),
                top_opps,
                roi.get("roi_percentage", 0) / 100,
                roi.get("annual_savings_usd", 0),
                roi.get("payback_months", 0),
                app.get("business_criticality", ""),
                app.get("department", "")
            ]
            
            for col, value in enumerate(data, 1):
                cell = ws.cell(row=row, column=col, value=value)
                cell.border = self.border
                
                if col == 3:  # Automation score
                    cell.number_format = '0.00'
                elif col == 5:  # ROI
                    cell.number_format = self.percent_format
                elif col == 6:  # Savings
                    cell.number_format = self.money_format
        
        # Column widths
        widths = [40, 22, 16, 35, 10, 16, 12, 12, 20]
        for i, width in enumerate(widths, 1):
            ws.column_dimensions[get_column_letter(i)].width = width
        
        ws.freeze_panes = 'A5'
    
    def _create_category_analysis(self, wb: Workbook, all_apps: list):
        """Create category-level analysis sheet"""
        ws = wb.create_sheet("Category Analysis")
        
        # Title
        ws.merge_cells('A1:H1')
        ws['A1'] = "Automation Analysis by Application Category"
        ws['A1'].font = Font(bold=True, size=16, color="1F4E79")
        
        # Aggregate by category
        categories = {}
        for app in all_apps:
            cat = app.get("category", "Unknown")
            if cat not in categories:
                categories[cat] = {
                    "count": 0,
                    "total_cost": 0,
                    "total_savings": 0,
                    "total_impl": 0,
                    "roi_sum": 0,
                    "auto_sum": 0
                }
            categories[cat]["count"] += 1
            categories[cat]["total_cost"] += app.get("annual_cost_usd", 0)
            categories[cat]["total_savings"] += app.get("roi_analysis", {}).get("annual_savings_usd", 0)
            categories[cat]["total_impl"] += app.get("roi_analysis", {}).get("implementation_cost_usd", 0)
            categories[cat]["roi_sum"] += app.get("roi_analysis", {}).get("roi_percentage", 0)
            categories[cat]["auto_sum"] += app.get("automation_potential", {}).get("overall_score", 0)
        
        # Headers
        headers = [
            "Category", "App Count", "Annual IT Spend", "Potential Savings",
            "Implementation Cost", "Avg ROI %", "Avg Automation Score", "Priority Rank"
        ]
        
        for col, header in enumerate(headers, 1):
            cell = ws.cell(row=3, column=col, value=header)
            cell.font = self.header_font
            cell.fill = self.header_fill
            cell.border = self.border
        
        # Sort categories by savings potential
        sorted_cats = sorted(
            categories.items(),
            key=lambda x: x[1]["total_savings"],
            reverse=True
        )
        
        for idx, (cat, data) in enumerate(sorted_cats, 1):
            row = idx + 3
            avg_roi = data["roi_sum"] / data["count"]
            avg_auto = data["auto_sum"] / data["count"]
            
            values = [
                cat,
                data["count"],
                data["total_cost"],
                data["total_savings"],
                data["total_impl"],
                avg_roi / 100,
                avg_auto,
                idx
            ]
            
            for col, value in enumerate(values, 1):
                cell = ws.cell(row=row, column=col, value=value)
                cell.border = self.border
                
                if col in [3, 4, 5]:
                    cell.number_format = self.money_format
                elif col == 6:
                    cell.number_format = self.percent_format
                elif col == 7:
                    cell.number_format = '0.00'
        
        # Column widths
        widths = [25, 12, 18, 18, 18, 12, 18, 14]
        for i, width in enumerate(widths, 1):
            ws.column_dimensions[get_column_letter(i)].width = width
        
        # Add bar chart for savings by category
        chart = BarChart()
        chart.type = "col"
        chart.style = 10
        chart.title = "Potential Savings by Category"
        chart.y_axis.title = "Savings ($)"
        chart.x_axis.title = "Category"
        
        data_ref = Reference(ws, min_col=4, min_row=3, max_row=3 + len(categories), max_col=4)
        cats_ref = Reference(ws, min_col=1, min_row=4, max_row=3 + len(categories))
        chart.add_data(data_ref, titles_from_data=True)
        chart.set_categories(cats_ref)
        chart.width = 18
        chart.height = 10
        
        ws.add_chart(chart, "A" + str(5 + len(categories)))
    
    def _create_department_analysis(self, wb: Workbook, all_apps: list):
        """Create department-level analysis sheet"""
        ws = wb.create_sheet("Department Analysis")
        
        # Title
        ws.merge_cells('A1:G1')
        ws['A1'] = "Automation Analysis by Department"
        ws['A1'].font = Font(bold=True, size=16, color="1F4E79")
        
        # Aggregate by department
        departments = {}
        for app in all_apps:
            dept = app.get("department", "Unknown")
            if dept not in departments:
                departments[dept] = {
                    "count": 0,
                    "total_savings": 0,
                    "avg_roi": 0,
                    "roi_sum": 0,
                    "high_priority": 0
                }
            departments[dept]["count"] += 1
            departments[dept]["total_savings"] += app.get("roi_analysis", {}).get("annual_savings_usd", 0)
            departments[dept]["roi_sum"] += app.get("roi_analysis", {}).get("roi_percentage", 0)
            if app.get("automation_potential", {}).get("overall_score", 0) > 0.7:
                departments[dept]["high_priority"] += 1
        
        # Headers
        headers = [
            "Department", "Total Apps", "Potential Savings", 
            "Avg ROI %", "High Priority Apps", "Recommendation"
        ]
        
        for col, header in enumerate(headers, 1):
            cell = ws.cell(row=3, column=col, value=header)
            cell.font = self.header_font
            cell.fill = self.header_fill
            cell.border = self.border
        
        # Sort by savings
        sorted_depts = sorted(
            departments.items(),
            key=lambda x: x[1]["total_savings"],
            reverse=True
        )
        
        for idx, (dept, data) in enumerate(sorted_depts, 1):
            row = idx + 3
            avg_roi = data["roi_sum"] / data["count"]
            
            # Generate recommendation
            if avg_roi > 60 and data["high_priority"] > 2:
                recommendation = "🟢 High Priority - Immediate Focus"
            elif avg_roi > 40:
                recommendation = "🟡 Medium Priority - Plan for Q2"
            else:
                recommendation = "🔵 Low Priority - Monitor"
            
            values = [
                dept,
                data["count"],
                data["total_savings"],
                avg_roi / 100,
                data["high_priority"],
                recommendation
            ]
            
            for col, value in enumerate(values, 1):
                cell = ws.cell(row=row, column=col, value=value)
                cell.border = self.border
                
                if col == 3:
                    cell.number_format = self.money_format
                elif col == 4:
                    cell.number_format = self.percent_format
        
        # Column widths
        widths = [22, 12, 18, 12, 18, 35]
        for i, width in enumerate(widths, 1):
            ws.column_dimensions[get_column_letter(i)].width = width
    
    def _create_implementation_roadmap(self, wb: Workbook, top_roi: list, high_automation: list):
        """Create implementation roadmap sheet"""
        ws = wb.create_sheet("Implementation Roadmap")
        
        # Title
        ws.merge_cells('A1:G1')
        ws['A1'] = "Gen AI Automation Implementation Roadmap"
        ws['A1'].font = Font(bold=True, size=16, color="1F4E79")
        
        # Phase definitions
        phases = [
            {
                "name": "Phase 1: Quick Wins (0-6 months)",
                "color": "70AD47",
                "criteria": "ROI > 75%, Payback < 6 months, Non-Critical",
                "apps": []
            },
            {
                "name": "Phase 2: Strategic Automation (6-12 months)",
                "color": "4472C4",
                "criteria": "ROI > 50%, Automation Score > 0.6",
                "apps": []
            },
            {
                "name": "Phase 3: Enterprise Scale (12-24 months)",
                "color": "7030A0",
                "criteria": "High value, complex integration",
                "apps": []
            }
        ]
        
        # Categorize apps into phases
        for app in high_automation:
            roi = app.get("roi_analysis", {})
            auto = app.get("automation_potential", {})
            crit = app.get("business_criticality", "")
            
            roi_pct = roi.get("roi_percentage", 0)
            payback = roi.get("payback_months", 100)
            auto_score = auto.get("overall_score", 0)
            
            if roi_pct > 75 and payback < 6 and crit not in ["Critical", "High"]:
                phases[0]["apps"].append(app)
            elif roi_pct > 50 and auto_score > 0.6:
                phases[1]["apps"].append(app)
            else:
                phases[2]["apps"].append(app)
        
        current_row = 3
        for phase in phases:
            # Phase header
            ws.merge_cells(f'A{current_row}:G{current_row}')
            cell = ws[f'A{current_row}']
            cell.value = phase["name"]
            cell.font = Font(bold=True, size=12, color="FFFFFF")
            cell.fill = PatternFill(start_color=phase["color"], end_color=phase["color"], fill_type="solid")
            
            current_row += 1
            ws[f'A{current_row}'] = f"Criteria: {phase['criteria']}"
            ws[f'A{current_row}'].font = Font(italic=True, size=9)
            
            current_row += 1
            # Column headers
            headers = ["Application", "Category", "ROI %", "Annual Savings", "Impl. Cost", "Payback", "Risk Level"]
            for col, header in enumerate(headers, 1):
                cell = ws.cell(row=current_row, column=col, value=header)
                cell.font = Font(bold=True)
                cell.fill = self.subheader_fill
                cell.font = Font(bold=True, color="FFFFFF")
                cell.border = self.border
            
            current_row += 1
            
            # Add apps (limit to 10 per phase)
            for app in phase["apps"][:10]:
                roi = app.get("roi_analysis", {})
                crit = app.get("business_criticality", "Low")
                risk = "Low" if crit == "Low" else "Medium" if crit == "Medium" else "High"
                
                values = [
                    app.get("name", "")[:40],
                    app.get("category", ""),
                    roi.get("roi_percentage", 0) / 100,
                    roi.get("annual_savings_usd", 0),
                    roi.get("implementation_cost_usd", 0),
                    f"{roi.get('payback_months', 0):.1f} mo",
                    risk
                ]
                
                for col, value in enumerate(values, 1):
                    cell = ws.cell(row=current_row, column=col, value=value)
                    cell.border = self.border
                    
                    if col == 3:
                        cell.number_format = self.percent_format
                    elif col in [4, 5]:
                        cell.number_format = self.money_format
                
                current_row += 1
            
            # Phase summary
            if phase["apps"]:
                total_savings = sum(a.get("roi_analysis", {}).get("annual_savings_usd", 0) for a in phase["apps"][:10])
                total_cost = sum(a.get("roi_analysis", {}).get("implementation_cost_usd", 0) for a in phase["apps"][:10])
                
                current_row += 1
                ws.cell(row=current_row, column=3, value="Phase Total:").font = Font(bold=True)
                ws.cell(row=current_row, column=4, value=total_savings).number_format = self.money_format
                ws.cell(row=current_row, column=4).font = Font(bold=True)
                ws.cell(row=current_row, column=5, value=total_cost).number_format = self.money_format
                ws.cell(row=current_row, column=5).font = Font(bold=True)
            
            current_row += 2
        
        # Column widths
        widths = [40, 22, 10, 16, 16, 12, 12]
        for i, width in enumerate(widths, 1):
            ws.column_dimensions[get_column_letter(i)].width = width
    
    def _create_full_portfolio(self, wb: Workbook, all_apps: list):
        """Create full portfolio data sheet"""
        ws = wb.create_sheet("Full Portfolio Data")
        
        # Title
        ws['A1'] = "Complete Application Portfolio"
        ws['A1'].font = Font(bold=True, size=14)
        
        # Convert to DataFrame
        df_data = []
        for app in all_apps:
            roi = app.get("roi_analysis", {})
            auto = app.get("automation_potential", {})
            
            df_data.append({
                "App ID": app.get("app_id", ""),
                "Name": app.get("name", ""),
                "Category": app.get("category", ""),
                "Subcategory": app.get("subcategory", ""),
                "Department": app.get("department", ""),
                "Criticality": app.get("business_criticality", ""),
                "Lifecycle": app.get("lifecycle_stage", ""),
                "Vendor": app.get("vendor", ""),
                "Annual Cost": app.get("annual_cost_usd", 0),
                "User Count": app.get("user_count", 0),
                "Daily Transactions": app.get("transaction_volume_daily", 0),
                "Automation Score": auto.get("overall_score", 0),
                "ROI %": roi.get("roi_percentage", 0),
                "Annual Savings": roi.get("annual_savings_usd", 0),
                "Implementation Cost": roi.get("implementation_cost_usd", 0),
                "Payback Months": roi.get("payback_months", 0),
                "3-Year NPV": roi.get("net_present_value_3yr", 0),
                "Technology": app.get("technology_stack", {}).get("primary", ""),
                "Infrastructure": app.get("technology_stack", {}).get("infrastructure", ""),
                "Data Classification": app.get("data_classification", ""),
            })
        
        df = pd.DataFrame(df_data)
        
        # Write headers
        for col, column_name in enumerate(df.columns, 1):
            cell = ws.cell(row=3, column=col, value=column_name)
            cell.font = self.header_font
            cell.fill = self.header_fill
            cell.border = self.border
        
        # Write data
        for row_idx, row_data in enumerate(df.values, 4):
            for col_idx, value in enumerate(row_data, 1):
                cell = ws.cell(row=row_idx, column=col_idx, value=value)
                cell.border = self.border
                
                # Format specific columns
                col_name = df.columns[col_idx - 1]
                if col_name in ["Annual Cost", "Annual Savings", "Implementation Cost", "3-Year NPV"]:
                    cell.number_format = self.money_format
                elif col_name == "ROI %":
                    cell.number_format = '0.0'
                elif col_name == "Automation Score":
                    cell.number_format = '0.00'
        
        # Auto-filter
        ws.auto_filter.ref = f"A3:{get_column_letter(len(df.columns))}{len(df) + 3}"
        
        # Freeze header
        ws.freeze_panes = 'A4'
        
        # Column widths
        for i, col in enumerate(df.columns, 1):
            ws.column_dimensions[get_column_letter(i)].width = max(12, len(str(col)) + 2)
        ws.column_dimensions['B'].width = 45  # Name column
    
    def _create_roi_details(self, wb: Workbook, all_apps: list):
        """Create detailed ROI analysis sheet"""
        ws = wb.create_sheet("ROI Details")
        
        # Title
        ws['A1'] = "Detailed ROI Analysis"
        ws['A1'].font = Font(bold=True, size=14)
        
        # Summary statistics
        roi_values = [app.get("roi_analysis", {}).get("roi_percentage", 0) for app in all_apps]
        savings_values = [app.get("roi_analysis", {}).get("annual_savings_usd", 0) for app in all_apps]
        
        stats = [
            ("Total Applications", len(all_apps)),
            ("Average ROI", f"{sum(roi_values) / len(roi_values):.1f}%"),
            ("Median ROI", f"{sorted(roi_values)[len(roi_values)//2]:.1f}%"),
            ("Total Potential Savings", sum(savings_values)),
            ("Apps with ROI > 100%", len([r for r in roi_values if r > 100])),
            ("Apps with ROI 50-100%", len([r for r in roi_values if 50 <= r <= 100])),
            ("Apps with ROI 25-50%", len([r for r in roi_values if 25 <= r < 50])),
            ("Apps with ROI < 25%", len([r for r in roi_values if r < 25])),
        ]
        
        for idx, (label, value) in enumerate(stats, 3):
            ws.cell(row=idx, column=1, value=label).font = Font(bold=True)
            cell = ws.cell(row=idx, column=2, value=value)
            if "Savings" in label:
                cell.number_format = self.money_format
        
        # ROI distribution data for chart
        row_start = 15
        ws.cell(row=row_start, column=1, value="ROI Range").font = Font(bold=True)
        ws.cell(row=row_start, column=2, value="Count").font = Font(bold=True)
        
        roi_ranges = [
            ("< 0%", len([r for r in roi_values if r < 0])),
            ("0-25%", len([r for r in roi_values if 0 <= r < 25])),
            ("25-50%", len([r for r in roi_values if 25 <= r < 50])),
            ("50-75%", len([r for r in roi_values if 50 <= r < 75])),
            ("75-100%", len([r for r in roi_values if 75 <= r < 100])),
            ("> 100%", len([r for r in roi_values if r >= 100])),
        ]
        
        for idx, (range_label, count) in enumerate(roi_ranges, row_start + 1):
            ws.cell(row=idx, column=1, value=range_label)
            ws.cell(row=idx, column=2, value=count)
        
        # Add pie chart
        pie = PieChart()
        pie.title = "ROI Distribution"
        labels = Reference(ws, min_col=1, min_row=row_start + 1, max_row=row_start + len(roi_ranges))
        data = Reference(ws, min_col=2, min_row=row_start, max_row=row_start + len(roi_ranges))
        pie.add_data(data, titles_from_data=True)
        pie.set_categories(labels)
        pie.width = 12
        pie.height = 10
        
        ws.add_chart(pie, "D3")
        
        ws.column_dimensions['A'].width = 25
        ws.column_dimensions['B'].width = 15


def generate_report(output_path: str = None) -> str:
    """Convenience function to generate report"""
    generator = ExcelReportGenerator()
    return generator.generate_report(output_path)


if __name__ == "__main__":
    print("[*] Generating Excel Report...")
    path = generate_report()
    print(f"[OK] Report generated: {path}")

