"""
Push 1000 Business Applications to ServiceNow
Reads credentials from .env file
"""
import json
import requests
from requests.auth import HTTPBasicAuth
import time
import os

from config import get_settings

# Get settings from .env
settings = get_settings()
INSTANCE_URL = settings.servicenow_instance_url
USERNAME = settings.servicenow_username
PASSWORD = settings.servicenow_password
TABLE = "cmdb_ci_business_app"


def load_or_generate_apps(count=1000):
    """Load existing mock apps or generate new ones"""
    mock_path = "data/servicenow_apps.json"
    
    if os.path.exists(mock_path):
        print(f"[*] Loading existing apps from {mock_path}...")
        with open(mock_path, "r", encoding="utf-8") as f:
            data = json.load(f)
        apps = data.get("applications", [])
        print(f"[OK] Loaded {len(apps)} applications")
        return apps
    else:
        print(f"[*] Generating {count} new applications...")
        from data.mock_app_generator import generate_all_applications, save_applications
        os.makedirs("data", exist_ok=True)
        apps = generate_all_applications(count)
        save_applications(apps, mock_path)
        return apps


def clear_existing_apps():
    """Delete all existing business apps from ServiceNow"""
    print("[*] Checking for existing business apps...")
    
    response = requests.get(
        f"{INSTANCE_URL}/api/now/table/{TABLE}",
        auth=HTTPBasicAuth(USERNAME, PASSWORD),
        headers={"Accept": "application/json"},
        params={"sysparm_fields": "sys_id", "sysparm_limit": 2000},
        timeout=60
    )
    
    if response.status_code == 200:
        apps = response.json().get("result", [])
        if apps:
            print(f"[*] Found {len(apps)} existing apps. Deleting...")
            for i, app in enumerate(apps, 1):
                requests.delete(
                    f"{INSTANCE_URL}/api/now/table/{TABLE}/{app['sys_id']}",
                    auth=HTTPBasicAuth(USERNAME, PASSWORD),
                    timeout=30
                )
                if i % 100 == 0:
                    print(f"    Deleted {i}/{len(apps)}...")
                time.sleep(0.05)
            print(f"[OK] Deleted {len(apps)} existing apps")
        else:
            print("[OK] No existing apps to delete")
    return True


def create_business_application(app_data):
    """Create a single business application in ServiceNow"""
    payload = {
        "name": app_data.get("name", "Unknown App")[:100],
        "short_description": app_data.get("description", "")[:1000],
        "comments": json.dumps({
            "app_id": app_data.get("app_id", ""),
            "category": app_data.get("category", ""),
            "subcategory": app_data.get("subcategory", ""),
            "department": app_data.get("department", ""),
            "vendor": app_data.get("vendor", ""),
            "technology_stack": app_data.get("technology_stack", {}),
            "automation_potential": app_data.get("automation_potential", {}),
            "roi_analysis": app_data.get("roi_analysis", {}),
            "user_count": app_data.get("user_count", 0),
            "transaction_volume_daily": app_data.get("transaction_volume_daily", 0),
            "data_classification": app_data.get("data_classification", ""),
            "regulatory_scope": app_data.get("regulatory_scope", []),
            "annual_cost_usd": app_data.get("annual_cost_usd", 0)
        })[:4000],
        "operational_status": "1" if app_data.get("lifecycle_stage") == "Production" else "2",
        "busines_criticality": {
            "Critical": "1 - most critical",
            "High": "2 - somewhat critical", 
            "Medium": "3 - less critical",
            "Low": "4 - not critical"
        }.get(app_data.get("business_criticality", "Medium"), "3 - less critical"),
        "used_for": app_data.get("category", ""),
        "cost": str(app_data.get("annual_cost_usd", 0)),
        "manufacturer": app_data.get("vendor", "In-House"),
        "number": app_data.get("app_id", ""),
    }
    
    try:
        response = requests.post(
            f"{INSTANCE_URL}/api/now/table/{TABLE}",
            auth=HTTPBasicAuth(USERNAME, PASSWORD),
            headers={"Accept": "application/json", "Content-Type": "application/json"},
            json=payload,
            timeout=30
        )
        return response.status_code in [200, 201]
    except:
        return False


def push_all_apps(apps, clear_first=False):
    """Push all applications to ServiceNow"""
    if clear_first:
        clear_existing_apps()
    
    print(f"\n[*] Pushing {len(apps)} business applications to ServiceNow...")
    print(f"    Instance: {INSTANCE_URL}")
    print(f"    Table: {TABLE}")
    print()
    
    success = 0
    failed = 0
    start_time = time.time()
    
    for i, app in enumerate(apps, 1):
        if create_business_application(app):
            success += 1
        else:
            failed += 1
        
        if i % 100 == 0:
            elapsed = time.time() - start_time
            rate = i / elapsed if elapsed > 0 else 0
            remaining = (len(apps) - i) / rate if rate > 0 else 0
            print(f"  {i}/{len(apps)} | Success: {success} | Failed: {failed} | ETA: {remaining:.0f}s")
        
        time.sleep(0.12)
    
    elapsed = time.time() - start_time
    
    print()
    print("=" * 60)
    print("PUSH COMPLETE")
    print("=" * 60)
    print(f"  Success: {success}")
    print(f"  Failed:  {failed}")
    print(f"  Time:    {elapsed:.1f} seconds")
    print()
    print(f"View apps: {INSTANCE_URL}/cmdb_ci_business_app_list.do")
    print("=" * 60)
    
    return success, failed


if __name__ == "__main__":
    import argparse
    
    parser = argparse.ArgumentParser(description="Push business apps to ServiceNow")
    parser.add_argument("--clear", action="store_true", help="Clear existing apps first")
    parser.add_argument("--count", type=int, default=1000, help="Number of apps to push")
    args = parser.parse_args()
    
    print("=" * 60)
    print("ServiceNow Business Application Loader")
    print("=" * 60)
    print(f"Instance: {INSTANCE_URL}")
    print(f"Username: {USERNAME}")
    print()
    
    if not PASSWORD:
        print("[ERROR] SERVICENOW_PASSWORD not set in .env file!")
        print("        Copy env_template.txt to .env and fill in your credentials.")
        exit(1)
    
    apps = load_or_generate_apps(args.count)
    apps = apps[:args.count]
    
    push_all_apps(apps, clear_first=args.clear)
