"""
ServiceNow Gen AI Automation Analyzer
Main entry point for the application

This tool helps CIOs and CFOs identify applications suitable for Gen AI automation
and calculates potential ROI for automation initiatives.
"""

import os
import sys
import typer
from rich.console import Console
from rich.panel import Panel
from rich.markdown import Markdown
from rich.progress import Progress, SpinnerColumn, TextColumn
from rich.table import Table
from datetime import datetime

from config import get_settings, create_env_template
from servicenow_client import ServiceNowClient
from agent import ApplicationAnalyzerAgent
from report_generator import ExcelReportGenerator

app = typer.Typer(
    name="servicenow-ai-analyzer",
    help="🤖 Analyze ServiceNow applications for Gen AI automation opportunities",
    add_completion=False
)
console = Console()


def print_banner():
    """Print application banner"""
    banner = """
╔══════════════════════════════════════════════════════════════════════╗
║                                                                      ║
║   🤖  ServiceNow Gen AI Automation Analyzer                         ║
║                                                                      ║
║   Identify applications for automation & calculate ROI              ║
║   For CIOs and CFOs - Data-driven automation decisions              ║
║                                                                      ║
╚══════════════════════════════════════════════════════════════════════╝
"""
    console.print(banner, style="bold cyan")


@app.command()
def generate_data(
    count: int = typer.Option(1000, "--count", "-n", help="Number of applications to generate"),
    output: str = typer.Option("data/servicenow_apps.json", "--output", "-o", help="Output file path")
):
    """
    Generate mock ServiceNow application data for testing.
    Creates realistic enterprise applications for a financial services company.
    """
    print_banner()
    
    console.print("\n[bold yellow]📊 Generating Mock ServiceNow Data[/bold yellow]\n")
    
    # Ensure directory exists
    os.makedirs(os.path.dirname(output), exist_ok=True)
    
    with Progress(
        SpinnerColumn(),
        TextColumn("[progress.description]{task.description}"),
        console=console
    ) as progress:
        task = progress.add_task("Generating applications...", total=None)
        
        from data.mock_app_generator import generate_all_applications, save_applications
        
        apps = generate_all_applications(count)
        save_applications(apps, output)
        
        progress.update(task, completed=True)
    
    console.print(f"\n[green]✅ Generated {count} applications to {output}[/green]")
    
    # Show sample
    console.print("\n[bold]Sample Application Categories:[/bold]")
    categories = {}
    for app in apps:
        cat = app.get("category", "Unknown")
        categories[cat] = categories.get(cat, 0) + 1
    
    table = Table(title="Applications by Category")
    table.add_column("Category", style="cyan")
    table.add_column("Count", justify="right")
    
    for cat, count in sorted(categories.items(), key=lambda x: -x[1]):
        table.add_row(cat, str(count))
    
    console.print(table)


@app.command()
def analyze(
    query: str = typer.Option(
        None, 
        "--query", "-q", 
        help="Custom analysis query (uses default if not provided)"
    ),
    output_excel: bool = typer.Option(
        True, 
        "--excel/--no-excel", 
        help="Generate Excel report"
    ),
    output_path: str = typer.Option(
        None, 
        "--output", "-o", 
        help="Custom output path for Excel report"
    )
):
    """
    Run the AI-powered portfolio analysis.
    Analyzes applications and generates automation recommendations.
    """
    print_banner()
    settings = get_settings()
    
    console.print("\n[bold cyan]🔍 Starting Portfolio Analysis[/bold cyan]\n")
    
    # Check for mock data
    if settings.use_mock_data:
        mock_path = settings.mock_data_path
        if not os.path.exists(mock_path) and not os.path.exists(f"data/{mock_path}"):
            console.print("[yellow]⚠️  Mock data not found. Generating now...[/yellow]\n")
            from data.mock_app_generator import generate_all_applications, save_applications
            os.makedirs("data", exist_ok=True)
            apps = generate_all_applications(1000)
            save_applications(apps, "data/servicenow_apps.json")
    
    # Initialize and run agent
    with Progress(
        SpinnerColumn(),
        TextColumn("[progress.description]{task.description}"),
        console=console
    ) as progress:
        task = progress.add_task("Initializing AI agent...", total=None)
        
        try:
            agent = ApplicationAnalyzerAgent()
            progress.update(task, description="Running portfolio analysis...")
            
            result = agent.run(query)
            
            progress.update(task, completed=True)
        except Exception as e:
            console.print(f"\n[red]❌ Error during analysis: {str(e)}[/red]")
            raise typer.Exit(1)
    
    # Display results
    console.print("\n" + "=" * 70)
    console.print("[bold green]📋 ANALYSIS RESULTS[/bold green]")
    console.print("=" * 70 + "\n")
    
    from langchain_core.messages import AIMessage, HumanMessage
    
    for msg in result.get("messages", []):
        if isinstance(msg, AIMessage):
            console.print(Markdown(msg.content))
    
    if result.get("error"):
        console.print(f"\n[red]⚠️  Warning: {result['error']}[/red]")
    
    # Generate Excel report
    if output_excel:
        console.print("\n[bold cyan]📊 Generating Excel Report...[/bold cyan]")
        
        try:
            generator = ExcelReportGenerator()
            report_path = generator.generate_report(output_path)
            
            console.print(Panel(
                f"[bold green]Excel report saved to:[/bold green]\n{report_path}",
                title="📈 Report Generated",
                border_style="green"
            ))
        except Exception as e:
            console.print(f"\n[red]❌ Error generating report: {str(e)}[/red]")


@app.command()
def report(
    output: str = typer.Option(
        None, 
        "--output", "-o", 
        help="Output path for the Excel report"
    )
):
    """
    Generate only the Excel report without running AI analysis.
    Uses existing data from ServiceNow or mock data.
    """
    print_banner()
    settings = get_settings()
    
    console.print("\n[bold cyan]📊 Generating Excel Report[/bold cyan]\n")
    
    # Check for mock data
    if settings.use_mock_data:
        mock_paths = [
            settings.mock_data_path,
            "data/servicenow_apps.json",
            os.path.join(os.path.dirname(__file__), "data", "servicenow_apps.json")
        ]
        
        found = False
        for path in mock_paths:
            if os.path.exists(path):
                found = True
                break
        
        if not found:
            console.print("[yellow]⚠️  Mock data not found. Generating now...[/yellow]\n")
            from data.mock_app_generator import generate_all_applications, save_applications
            os.makedirs("data", exist_ok=True)
            apps = generate_all_applications(1000)
            save_applications(apps, "data/servicenow_apps.json")
    
    with Progress(
        SpinnerColumn(),
        TextColumn("[progress.description]{task.description}"),
        console=console
    ) as progress:
        task = progress.add_task("Generating report...", total=None)
        
        generator = ExcelReportGenerator()
        report_path = generator.generate_report(output)
        
        progress.update(task, completed=True)
    
    console.print(Panel(
        f"[bold green]✅ Report generated successfully![/bold green]\n\n"
        f"📁 Location: {report_path}\n\n"
        f"The report includes:\n"
        f"  • Executive Summary\n"
        f"  • Top 50 Recommendations\n"
        f"  • Automation Candidates\n"
        f"  • Category Analysis\n"
        f"  • Department Analysis\n"
        f"  • Implementation Roadmap\n"
        f"  • Full Portfolio Data\n"
        f"  • ROI Details",
        title="📊 Excel Report",
        border_style="green"
    ))


@app.command()
def summary():
    """
    Show a quick summary of the application portfolio.
    """
    print_banner()
    
    console.print("\n[bold cyan]📋 Portfolio Summary[/bold cyan]\n")
    
    try:
        client = ServiceNowClient()
        summary = client.get_portfolio_summary()
        
        # Main stats
        table = Table(title="Portfolio Overview")
        table.add_column("Metric", style="cyan")
        table.add_column("Value", justify="right", style="green")
        
        table.add_row("Total Applications", f"{summary['total_applications']:,}")
        table.add_row("Total Annual IT Spend", f"${summary['total_annual_cost_usd']:,.0f}")
        table.add_row("Total Users", f"{summary['total_user_count']:,}")
        
        console.print(table)
        
        # By criticality
        console.print("\n[bold]Applications by Criticality:[/bold]")
        crit_table = Table()
        crit_table.add_column("Criticality")
        crit_table.add_column("Count", justify="right")
        
        for crit, count in sorted(summary['by_criticality'].items()):
            crit_table.add_row(crit, str(count))
        
        console.print(crit_table)
        
        # By category
        console.print("\n[bold]Applications by Category:[/bold]")
        cat_table = Table()
        cat_table.add_column("Category", style="cyan")
        cat_table.add_column("Count", justify="right")
        
        for cat, count in sorted(summary['by_category'].items(), key=lambda x: -x[1]):
            cat_table.add_row(cat, str(count))
        
        console.print(cat_table)
        
        # Top ROI opportunities
        console.print("\n[bold]Top 5 ROI Opportunities:[/bold]")
        top_roi = client.get_top_roi_opportunities(5)
        
        roi_table = Table()
        roi_table.add_column("Application")
        roi_table.add_column("ROI %", justify="right")
        roi_table.add_column("Annual Savings", justify="right")
        roi_table.add_column("Payback", justify="right")
        
        for app in top_roi:
            roi = app.get("roi_analysis", {})
            roi_table.add_row(
                app.get("name", "")[:35],
                f"{roi.get('roi_percentage', 0):.0f}%",
                f"${roi.get('annual_savings_usd', 0):,.0f}",
                f"{roi.get('payback_months', 0):.1f} mo"
            )
        
        console.print(roi_table)
        
    except Exception as e:
        console.print(f"[red]❌ Error: {str(e)}[/red]")
        console.print("[yellow]💡 Tip: Run 'python main.py generate-data' first to create mock data[/yellow]")


@app.command()
def init():
    """
    Initialize the application with configuration templates.
    """
    print_banner()
    
    console.print("\n[bold cyan]🔧 Initializing Application[/bold cyan]\n")
    
    # Create directories
    os.makedirs("data", exist_ok=True)
    os.makedirs("output", exist_ok=True)
    
    # Create .env template
    create_env_template()
    
    console.print(Panel(
        "[bold green]✅ Initialization complete![/bold green]\n\n"
        "Created:\n"
        "  📁 data/ - For application data\n"
        "  📁 output/ - For generated reports\n"
        "  📄 .env - Configuration file (edit with your settings)\n\n"
        "[bold yellow]Next Steps:[/bold yellow]\n"
        "  1. Edit .env with your ServiceNow credentials\n"
        "  2. Add your OpenAI API key for AI analysis\n"
        "  3. Run: python main.py generate-data\n"
        "  4. Run: python main.py analyze",
        title="🚀 Setup Complete",
        border_style="green"
    ))


@app.command()
def top(
    n: int = typer.Option(10, "--count", "-n", help="Number of recommendations to show"),
    by: str = typer.Option("roi", "--by", "-b", help="Sort by: roi, automation, savings")
):
    """
    Show top N automation recommendations quickly.
    """
    print_banner()
    
    console.print(f"\n[bold cyan]🏆 Top {n} Recommendations (by {by})[/bold cyan]\n")
    
    try:
        client = ServiceNowClient()
        
        if by == "automation":
            apps = client.get_high_automation_candidates(0.0)[:n]
            sort_key = lambda x: x.get("automation_potential", {}).get("overall_score", 0)
        elif by == "savings":
            apps = client.get_top_roi_opportunities(n * 2)
            apps.sort(key=lambda x: x.get("roi_analysis", {}).get("annual_savings_usd", 0), reverse=True)
            apps = apps[:n]
            sort_key = lambda x: x.get("roi_analysis", {}).get("annual_savings_usd", 0)
        else:  # roi
            apps = client.get_top_roi_opportunities(n)
            sort_key = lambda x: x.get("roi_analysis", {}).get("roi_percentage", 0)
        
        table = Table(title=f"Top {n} Applications for Gen AI Automation")
        table.add_column("#", style="dim", width=4)
        table.add_column("Application", style="cyan", width=40)
        table.add_column("Category", width=20)
        table.add_column("ROI %", justify="right", style="green")
        table.add_column("Annual Savings", justify="right")
        table.add_column("Auto Score", justify="right")
        table.add_column("Payback", justify="right")
        
        for idx, app in enumerate(apps, 1):
            roi = app.get("roi_analysis", {})
            auto = app.get("automation_potential", {})
            
            table.add_row(
                str(idx),
                app.get("name", "")[:38],
                app.get("category", "")[:18],
                f"{roi.get('roi_percentage', 0):.0f}%",
                f"${roi.get('annual_savings_usd', 0):,.0f}",
                f"{auto.get('overall_score', 0):.2f}",
                f"{roi.get('payback_months', 0):.1f} mo"
            )
        
        console.print(table)
        
        # Summary
        total_savings = sum(a.get("roi_analysis", {}).get("annual_savings_usd", 0) for a in apps)
        console.print(f"\n[bold]Total Potential Savings from Top {n}: [green]${total_savings:,.0f}[/green] per year[/bold]")
        
    except Exception as e:
        console.print(f"[red]❌ Error: {str(e)}[/red]")


@app.callback()
def main():
    """
    ServiceNow Gen AI Automation Analyzer
    
    Analyze your application portfolio to identify automation opportunities
    and calculate ROI for Gen AI initiatives.
    """
    pass


if __name__ == "__main__":
    app()

