"""
Enhanced Excel Report Generator with AI Insights
Includes GPT-4 analysis as a dedicated sheet
"""

import os
import re
from datetime import datetime
from typing import Any
import pandas as pd
from openpyxl import Workbook
from openpyxl.styles import Font, PatternFill, Alignment, Border, Side
from openpyxl.chart import BarChart, PieChart, Reference
from openpyxl.utils.dataframe import dataframe_to_rows
from openpyxl.utils import get_column_letter

from config import get_settings
from servicenow_client import ServiceNowClient


class EnhancedReportGenerator:
    """Generates Excel reports with AI insights for CIO/CFO"""
    
    def __init__(self, client: ServiceNowClient = None):
        self.client = client or ServiceNowClient()
        self.settings = get_settings()
        
        # Styles
        self.header_font = Font(bold=True, color="FFFFFF", size=11)
        self.header_fill = PatternFill(start_color="1F4E79", end_color="1F4E79", fill_type="solid")
        self.ai_fill = PatternFill(start_color="7030A0", end_color="7030A0", fill_type="solid")
        self.highlight_fill = PatternFill(start_color="E2EFDA", end_color="E2EFDA", fill_type="solid")
        self.border = Border(
            left=Side(style='thin'), right=Side(style='thin'),
            top=Side(style='thin'), bottom=Side(style='thin')
        )
        self.money_format = '_("$"* #,##0_);_("$"* (#,##0);_("$"* "-"_);_(@_)'
        self.percent_format = '0.0%'
    
    def generate_report(self, output_path: str = None, ai_insights: str = None) -> str:
        """Generate the complete Excel report with AI insights"""
        if output_path is None:
            os.makedirs(self.settings.output_directory, exist_ok=True)
            timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
            output_path = os.path.join(
                self.settings.output_directory, 
                f"AI_Automation_Analysis_{timestamp}.xlsx"
            )
        
        # Fetch data
        print("[*] Fetching application data from ServiceNow...")
        all_apps = self.client.get_all_applications()
        summary = self.client.get_portfolio_summary()
        top_roi = self.client.get_top_roi_opportunities(100)
        high_automation = self.client.get_high_automation_candidates(0.5)
        
        # Create workbook
        wb = Workbook()
        
        # Create sheets (AI Insights first!)
        if ai_insights:
            self._create_ai_insights_sheet(wb, ai_insights, summary)
        self._create_executive_summary(wb, summary, all_apps)
        self._create_top_recommendations(wb, top_roi)
        self._create_automation_candidates(wb, high_automation)
        self._create_category_analysis(wb, all_apps)
        self._create_department_analysis(wb, all_apps)
        self._create_implementation_roadmap(wb, top_roi, high_automation)
        self._create_full_portfolio(wb, all_apps)
        
        # Remove default sheet
        if "Sheet" in wb.sheetnames:
            del wb["Sheet"]
        
        wb.save(output_path)
        print(f"[OK] Report saved to: {output_path}")
        
        return output_path
    
    def _create_ai_insights_sheet(self, wb: Workbook, ai_insights: str, summary: dict):
        """Create AI Insights sheet with GPT-4 recommendations"""
        ws = wb.create_sheet("AI Insights", 0)
        
        # Title
        ws.merge_cells('A1:L1')
        ws['A1'] = "AI-Powered Automation Recommendations"
        ws['A1'].font = Font(bold=True, size=20, color="FFFFFF")
        ws['A1'].fill = self.ai_fill
        ws['A1'].alignment = Alignment(horizontal='center')
        
        # Subtitle
        ws.merge_cells('A2:L2')
        ws['A2'] = f"Generated by GPT-4 | {datetime.now().strftime('%B %d, %Y at %I:%M %p')} | {summary['total_applications']} Applications Analyzed"
        ws['A2'].font = Font(italic=True, size=11, color="666666")
        ws['A2'].alignment = Alignment(horizontal='center')
        
        # AI Response
        row = 4
        ws.merge_cells(f'A{row}:L{row}')
        ws[f'A{row}'] = "Executive AI Analysis"
        ws[f'A{row}'].font = Font(bold=True, size=14)
        ws[f'A{row}'].fill = self.header_fill
        ws[f'A{row}'].font = self.header_font
        
        row += 2
        
        # Parse and format AI response
        lines = ai_insights.split('\n')
        for line in lines:
            line = line.strip()
            if not line:
                row += 1
                continue
            
            # Handle headers
            if line.startswith('##') or line.startswith('**') and line.endswith('**'):
                ws.merge_cells(f'A{row}:L{row}')
                clean_line = line.replace('#', '').replace('**', '').strip()
                ws[f'A{row}'] = clean_line
                ws[f'A{row}'].font = Font(bold=True, size=12, color="1F4E79")
                row += 1
            # Handle bullet points
            elif line.startswith('-') or line.startswith('*') or line.startswith('•'):
                ws.merge_cells(f'A{row}:L{row}')
                ws[f'A{row}'] = "  " + line
                ws[f'A{row}'].font = Font(size=10)
                row += 1
            # Handle numbered items
            elif len(line) > 2 and line[0].isdigit() and line[1] in '.):':
                ws.merge_cells(f'A{row}:L{row}')
                ws[f'A{row}'] = line
                ws[f'A{row}'].font = Font(size=10)
                row += 1
            else:
                ws.merge_cells(f'A{row}:L{row}')
                ws[f'A{row}'] = line
                ws[f'A{row}'].font = Font(size=10)
                row += 1
        
        # Key metrics box
        row += 2
        ws.merge_cells(f'A{row}:D{row}')
        ws[f'A{row}'] = "Key Portfolio Metrics"
        ws[f'A{row}'].font = self.header_font
        ws[f'A{row}'].fill = self.ai_fill
        
        row += 1
        metrics = [
            ("Total Applications Analyzed", summary['total_applications']),
            ("Data Source", summary.get('data_source', 'ServiceNow').upper()),
            ("Total Annual IT Spend", f"${summary['total_annual_cost_usd']:,.0f}"),
            ("Analysis Date", datetime.now().strftime('%Y-%m-%d')),
        ]
        
        for label, value in metrics:
            ws[f'A{row}'] = label
            ws[f'A{row}'].font = Font(bold=True)
            ws[f'B{row}'] = str(value)
            ws[f'A{row}'].fill = self.highlight_fill
            ws[f'B{row}'].fill = self.highlight_fill
            row += 1
        
        # Column width
        ws.column_dimensions['A'].width = 80
        ws.column_dimensions['B'].width = 25
    
    def _create_executive_summary(self, wb: Workbook, summary: dict, all_apps: list):
        """Create executive summary sheet"""
        ws = wb.create_sheet("Executive Summary")
        
        ws.merge_cells('A1:H1')
        ws['A1'] = "Portfolio Overview - Executive Summary"
        ws['A1'].font = Font(bold=True, size=18, color="1F4E79")
        ws['A1'].alignment = Alignment(horizontal='center')
        
        ws.merge_cells('A2:H2')
        ws['A2'] = f"Data Source: ServiceNow | Generated: {datetime.now().strftime('%B %d, %Y')}"
        ws['A2'].font = Font(italic=True, size=10, color="666666")
        ws['A2'].alignment = Alignment(horizontal='center')
        
        # Key Metrics
        row = 4
        ws.merge_cells(f'A{row}:D{row}')
        ws[f'A{row}'] = "Portfolio Overview"
        ws[f'A{row}'].font = self.header_font
        ws[f'A{row}'].fill = self.header_fill
        
        row += 1
        total_savings = sum(app.get("roi_analysis", {}).get("annual_savings_usd", 0) for app in all_apps)
        total_impl = sum(app.get("roi_analysis", {}).get("implementation_cost_usd", 0) for app in all_apps)
        avg_roi = sum(app.get("roi_analysis", {}).get("roi_percentage", 0) for app in all_apps) / max(len(all_apps), 1)
        
        metrics = [
            ("Total Applications", f"{summary['total_applications']:,}"),
            ("Total Annual IT Spend", summary['total_annual_cost_usd']),
            ("Total Potential Savings", total_savings),
            ("Total Implementation Cost", total_impl),
            ("Average ROI", f"{avg_roi:.1f}%"),
            ("High Priority Apps (ROI>50%)", len([a for a in all_apps if a.get("roi_analysis", {}).get("roi_percentage", 0) > 50])),
        ]
        
        for label, value in metrics:
            ws[f'A{row}'] = label
            ws[f'A{row}'].font = Font(bold=True)
            if isinstance(value, (int, float)) and "%" not in str(label):
                ws[f'B{row}'] = value
                ws[f'B{row}'].number_format = self.money_format
            else:
                ws[f'B{row}'] = str(value)
            row += 1
        
        # Quick wins
        row += 2
        ws.merge_cells(f'A{row}:F{row}')
        ws[f'A{row}'] = "Top 5 Quick Wins (Fastest Payback)"
        ws[f'A{row}'].font = self.header_font
        ws[f'A{row}'].fill = PatternFill(start_color="70AD47", end_color="70AD47", fill_type="solid")
        
        row += 1
        headers = ["Application", "Category", "ROI %", "Annual Savings", "Payback"]
        for col, h in enumerate(headers, 1):
            ws.cell(row=row, column=col, value=h).font = Font(bold=True)
            ws.cell(row=row, column=col).border = self.border
        
        quick_wins = sorted(
            [a for a in all_apps if a.get("roi_analysis", {}).get("payback_months", 100) < 12],
            key=lambda x: x.get("roi_analysis", {}).get("payback_months", 100)
        )[:5]
        
        row += 1
        for app in quick_wins:
            roi = app.get("roi_analysis", {})
            ws.cell(row=row, column=1, value=app.get("name", "")[:40]).border = self.border
            ws.cell(row=row, column=2, value=app.get("category", "")).border = self.border
            ws.cell(row=row, column=3, value=roi.get("roi_percentage", 0)/100).number_format = self.percent_format
            ws.cell(row=row, column=3).border = self.border
            ws.cell(row=row, column=4, value=roi.get("annual_savings_usd", 0)).number_format = self.money_format
            ws.cell(row=row, column=4).border = self.border
            ws.cell(row=row, column=5, value=f"{roi.get('payback_months', 0):.1f} mo").border = self.border
            row += 1
        
        ws.column_dimensions['A'].width = 45
        ws.column_dimensions['B'].width = 25
        ws.column_dimensions['C'].width = 12
        ws.column_dimensions['D'].width = 18
        ws.column_dimensions['E'].width = 12
    
    def _create_top_recommendations(self, wb: Workbook, top_roi: list):
        """Create top recommendations sheet"""
        ws = wb.create_sheet("Top 50 Recommendations")
        
        ws.merge_cells('A1:J1')
        ws['A1'] = "Top 50 Applications by ROI Potential"
        ws['A1'].font = Font(bold=True, size=16, color="1F4E79")
        
        headers = ["Rank", "Application", "Category", "Criticality", "ROI %", 
                   "Annual Savings", "Impl Cost", "Payback", "3Y NPV", "Auto Score"]
        
        for col, h in enumerate(headers, 1):
            cell = ws.cell(row=3, column=col, value=h)
            cell.font = self.header_font
            cell.fill = self.header_fill
            cell.border = self.border
        
        for idx, app in enumerate(top_roi[:50], 1):
            row = idx + 3
            roi = app.get("roi_analysis", {})
            auto = app.get("automation_potential", {})
            
            data = [idx, app.get("name", "")[:42], app.get("category", ""),
                    app.get("business_criticality", ""), roi.get("roi_percentage", 0)/100,
                    roi.get("annual_savings_usd", 0), roi.get("implementation_cost_usd", 0),
                    roi.get("payback_months", 0), roi.get("net_present_value_3yr", 0),
                    auto.get("overall_score", 0)]
            
            for col, value in enumerate(data, 1):
                cell = ws.cell(row=row, column=col, value=value)
                cell.border = self.border
                if col == 5: cell.number_format = self.percent_format
                elif col in [6, 7, 9]: cell.number_format = self.money_format
                elif col == 10: cell.number_format = '0.00'
        
        widths = [6, 42, 22, 12, 10, 15, 15, 10, 15, 10]
        for i, w in enumerate(widths, 1):
            ws.column_dimensions[get_column_letter(i)].width = w
        
        ws.freeze_panes = 'A4'
    
    def _create_automation_candidates(self, wb: Workbook, candidates: list):
        """Create automation candidates sheet"""
        ws = wb.create_sheet("Automation Candidates")
        
        ws.merge_cells('A1:I1')
        ws['A1'] = "High Automation Potential Applications (Score >= 0.5)"
        ws['A1'].font = Font(bold=True, size=16, color="1F4E79")
        
        headers = ["Application", "Category", "Auto Score", "Top Opportunities",
                   "ROI %", "Annual Savings", "Payback", "Criticality", "Department"]
        
        for col, h in enumerate(headers, 1):
            cell = ws.cell(row=3, column=col, value=h)
            cell.font = self.header_font
            cell.fill = self.header_fill
            cell.border = self.border
        
        for idx, app in enumerate(candidates[:100], 1):
            row = idx + 3
            roi = app.get("roi_analysis", {})
            auto = app.get("automation_potential", {})
            opps = list(auto.get("opportunities", {}).keys())[:3]
            
            data = [app.get("name", "")[:40], app.get("category", ""),
                    auto.get("overall_score", 0), ", ".join(opps),
                    roi.get("roi_percentage", 0)/100, roi.get("annual_savings_usd", 0),
                    roi.get("payback_months", 0), app.get("business_criticality", ""),
                    app.get("department", "")]
            
            for col, value in enumerate(data, 1):
                cell = ws.cell(row=row, column=col, value=value)
                cell.border = self.border
                if col == 3: cell.number_format = '0.00'
                elif col == 5: cell.number_format = self.percent_format
                elif col == 6: cell.number_format = self.money_format
        
        widths = [40, 22, 12, 35, 10, 15, 10, 12, 20]
        for i, w in enumerate(widths, 1):
            ws.column_dimensions[get_column_letter(i)].width = w
        
        ws.freeze_panes = 'A4'
    
    def _create_category_analysis(self, wb: Workbook, all_apps: list):
        """Create category analysis sheet"""
        ws = wb.create_sheet("Category Analysis")
        
        ws.merge_cells('A1:H1')
        ws['A1'] = "Automation Analysis by Application Category"
        ws['A1'].font = Font(bold=True, size=16, color="1F4E79")
        
        # Aggregate
        categories = {}
        for app in all_apps:
            cat = app.get("category", "Unknown")
            if cat not in categories:
                categories[cat] = {"count": 0, "savings": 0, "impl": 0, "roi_sum": 0, "auto_sum": 0}
            categories[cat]["count"] += 1
            categories[cat]["savings"] += app.get("roi_analysis", {}).get("annual_savings_usd", 0)
            categories[cat]["impl"] += app.get("roi_analysis", {}).get("implementation_cost_usd", 0)
            categories[cat]["roi_sum"] += app.get("roi_analysis", {}).get("roi_percentage", 0)
            categories[cat]["auto_sum"] += app.get("automation_potential", {}).get("overall_score", 0)
        
        headers = ["Category", "Apps", "Annual Spend", "Potential Savings", "Impl Cost", "Avg ROI", "Avg Auto Score", "Priority"]
        for col, h in enumerate(headers, 1):
            cell = ws.cell(row=3, column=col, value=h)
            cell.font = self.header_font
            cell.fill = self.header_fill
            cell.border = self.border
        
        sorted_cats = sorted(categories.items(), key=lambda x: x[1]["savings"], reverse=True)
        
        for idx, (cat, data) in enumerate(sorted_cats, 1):
            row = idx + 3
            avg_roi = data["roi_sum"] / data["count"]
            avg_auto = data["auto_sum"] / data["count"]
            
            values = [cat, data["count"], data["savings"]*2, data["savings"], data["impl"], avg_roi/100, avg_auto, idx]
            
            for col, value in enumerate(values, 1):
                cell = ws.cell(row=row, column=col, value=value)
                cell.border = self.border
                if col in [3, 4, 5]: cell.number_format = self.money_format
                elif col == 6: cell.number_format = self.percent_format
                elif col == 7: cell.number_format = '0.00'
        
        widths = [28, 8, 18, 18, 18, 10, 14, 10]
        for i, w in enumerate(widths, 1):
            ws.column_dimensions[get_column_letter(i)].width = w
    
    def _create_department_analysis(self, wb: Workbook, all_apps: list):
        """Create department analysis sheet"""
        ws = wb.create_sheet("Department Analysis")
        
        ws.merge_cells('A1:G1')
        ws['A1'] = "Automation Analysis by Department"
        ws['A1'].font = Font(bold=True, size=16, color="1F4E79")
        
        departments = {}
        for app in all_apps:
            dept = app.get("department", "Unknown")
            if dept not in departments:
                departments[dept] = {"count": 0, "savings": 0, "roi_sum": 0, "high_priority": 0}
            departments[dept]["count"] += 1
            departments[dept]["savings"] += app.get("roi_analysis", {}).get("annual_savings_usd", 0)
            departments[dept]["roi_sum"] += app.get("roi_analysis", {}).get("roi_percentage", 0)
            if app.get("automation_potential", {}).get("overall_score", 0) > 0.7:
                departments[dept]["high_priority"] += 1
        
        headers = ["Department", "Total Apps", "Potential Savings", "Avg ROI", "High Priority Apps", "Recommendation"]
        for col, h in enumerate(headers, 1):
            cell = ws.cell(row=3, column=col, value=h)
            cell.font = self.header_font
            cell.fill = self.header_fill
            cell.border = self.border
        
        sorted_depts = sorted(departments.items(), key=lambda x: x[1]["savings"], reverse=True)
        
        for idx, (dept, data) in enumerate(sorted_depts, 1):
            row = idx + 3
            avg_roi = data["roi_sum"] / data["count"]
            
            if avg_roi > 60 and data["high_priority"] > 2:
                rec = "HIGH PRIORITY - Immediate Focus"
            elif avg_roi > 40:
                rec = "MEDIUM - Plan for Q2"
            else:
                rec = "MONITOR - Lower Priority"
            
            values = [dept, data["count"], data["savings"], avg_roi/100, data["high_priority"], rec]
            
            for col, value in enumerate(values, 1):
                cell = ws.cell(row=row, column=col, value=value)
                cell.border = self.border
                if col == 3: cell.number_format = self.money_format
                elif col == 4: cell.number_format = self.percent_format
        
        widths = [25, 12, 18, 12, 18, 30]
        for i, w in enumerate(widths, 1):
            ws.column_dimensions[get_column_letter(i)].width = w
    
    def _create_implementation_roadmap(self, wb: Workbook, top_roi: list, high_auto: list):
        """Create implementation roadmap sheet"""
        ws = wb.create_sheet("Implementation Roadmap")
        
        ws.merge_cells('A1:G1')
        ws['A1'] = "Gen AI Automation Implementation Roadmap"
        ws['A1'].font = Font(bold=True, size=16, color="1F4E79")
        
        phases = [
            {"name": "Phase 1: Quick Wins (0-6 months)", "color": "70AD47", "apps": []},
            {"name": "Phase 2: Strategic (6-12 months)", "color": "4472C4", "apps": []},
            {"name": "Phase 3: Enterprise (12-24 months)", "color": "7030A0", "apps": []},
        ]
        
        for app in high_auto:
            roi = app.get("roi_analysis", {})
            payback = roi.get("payback_months", 100)
            roi_pct = roi.get("roi_percentage", 0)
            crit = app.get("business_criticality", "")
            
            if payback < 6 and crit not in ["Critical"]:
                phases[0]["apps"].append(app)
            elif payback < 12 and roi_pct > 30:
                phases[1]["apps"].append(app)
            else:
                phases[2]["apps"].append(app)
        
        row = 3
        for phase in phases:
            ws.merge_cells(f'A{row}:G{row}')
            ws[f'A{row}'] = phase["name"]
            ws[f'A{row}'].font = Font(bold=True, size=12, color="FFFFFF")
            ws[f'A{row}'].fill = PatternFill(start_color=phase["color"], end_color=phase["color"], fill_type="solid")
            
            row += 1
            headers = ["Application", "Category", "ROI %", "Savings", "Cost", "Payback", "Risk"]
            for col, h in enumerate(headers, 1):
                ws.cell(row=row, column=col, value=h).font = Font(bold=True)
                ws.cell(row=row, column=col).border = self.border
            
            row += 1
            for app in phase["apps"][:8]:
                roi = app.get("roi_analysis", {})
                crit = app.get("business_criticality", "Low")
                risk = "Low" if crit == "Low" else "Medium" if crit == "Medium" else "High"
                
                values = [app.get("name", "")[:35], app.get("category", ""),
                          roi.get("roi_percentage", 0)/100, roi.get("annual_savings_usd", 0),
                          roi.get("implementation_cost_usd", 0), f"{roi.get('payback_months', 0):.1f} mo", risk]
                
                for col, value in enumerate(values, 1):
                    cell = ws.cell(row=row, column=col, value=value)
                    cell.border = self.border
                    if col == 3: cell.number_format = self.percent_format
                    elif col in [4, 5]: cell.number_format = self.money_format
                row += 1
            
            # Phase total
            if phase["apps"]:
                total_savings = sum(a.get("roi_analysis", {}).get("annual_savings_usd", 0) for a in phase["apps"][:8])
                ws.cell(row=row, column=3, value="TOTAL:").font = Font(bold=True)
                ws.cell(row=row, column=4, value=total_savings).number_format = self.money_format
                ws.cell(row=row, column=4).font = Font(bold=True)
            
            row += 2
        
        widths = [38, 22, 10, 15, 15, 12, 10]
        for i, w in enumerate(widths, 1):
            ws.column_dimensions[get_column_letter(i)].width = w
    
    def _create_full_portfolio(self, wb: Workbook, all_apps: list):
        """Create full portfolio data sheet"""
        ws = wb.create_sheet("Full Portfolio Data")
        
        ws['A1'] = "Complete Application Portfolio"
        ws['A1'].font = Font(bold=True, size=14)
        
        df_data = []
        for app in all_apps:
            roi = app.get("roi_analysis", {})
            auto = app.get("automation_potential", {})
            
            df_data.append({
                "App ID": app.get("app_id", ""),
                "Name": app.get("name", ""),
                "Category": app.get("category", ""),
                "Department": app.get("department", ""),
                "Criticality": app.get("business_criticality", ""),
                "Vendor": app.get("vendor", ""),
                "Annual Cost": app.get("annual_cost_usd", 0),
                "User Count": app.get("user_count", 0),
                "Auto Score": auto.get("overall_score", 0),
                "ROI %": roi.get("roi_percentage", 0),
                "Annual Savings": roi.get("annual_savings_usd", 0),
                "Impl Cost": roi.get("implementation_cost_usd", 0),
                "Payback Mo": roi.get("payback_months", 0),
                "3Y NPV": roi.get("net_present_value_3yr", 0),
            })
        
        df = pd.DataFrame(df_data)
        
        for col, column_name in enumerate(df.columns, 1):
            cell = ws.cell(row=3, column=col, value=column_name)
            cell.font = self.header_font
            cell.fill = self.header_fill
            cell.border = self.border
        
        for row_idx, row_data in enumerate(df.values, 4):
            for col_idx, value in enumerate(row_data, 1):
                cell = ws.cell(row=row_idx, column=col_idx, value=value)
                cell.border = self.border
                col_name = df.columns[col_idx - 1]
                if col_name in ["Annual Cost", "Annual Savings", "Impl Cost", "3Y NPV"]:
                    cell.number_format = self.money_format
                elif col_name == "Auto Score":
                    cell.number_format = '0.00'
        
        ws.auto_filter.ref = f"A3:{get_column_letter(len(df.columns))}{len(df) + 3}"
        ws.freeze_panes = 'A4'
        
        for i in range(1, len(df.columns) + 1):
            ws.column_dimensions[get_column_letter(i)].width = 14
        ws.column_dimensions['B'].width = 42


def generate_enhanced_report(ai_insights: str = None) -> str:
    """Convenience function"""
    generator = EnhancedReportGenerator()
    return generator.generate_report(ai_insights=ai_insights)


if __name__ == "__main__":
    print("[*] Generating Enhanced Report (without AI insights)...")
    path = generate_enhanced_report()
    print(f"[OK] Report: {path}")

