"""
Configuration management for ServiceNow AI Analyzer
Reads all settings from .env file
"""

import os
from pathlib import Path
from typing import Optional
from pydantic_settings import BaseSettings
from pydantic import Field
from dotenv import load_dotenv

# Load .env file from the project directory
env_path = Path(__file__).parent / ".env"
load_dotenv(env_path)


class Settings(BaseSettings):
    """Application settings loaded from environment variables"""
    
    # ServiceNow Configuration
    servicenow_instance_url: str = Field(
        default="https://dev201970.service-now.com",
        description="ServiceNow instance URL"
    )
    servicenow_username: str = Field(
        default="admin",
        description="ServiceNow username"
    )
    servicenow_password: str = Field(
        default="",
        description="ServiceNow password"
    )
    
    # OpenAI Configuration
    openai_api_key: str = Field(
        default="",
        description="OpenAI API key for LLM operations"
    )
    openai_model: str = Field(
        default="gpt-4o",
        description="OpenAI model to use"
    )
    
    # Application Settings
    use_mock_data: bool = Field(
        default=False,
        description="Use mock data instead of real ServiceNow API"
    )
    mock_data_path: str = Field(
        default="data/servicenow_apps.json",
        description="Path to mock data file"
    )
    output_directory: str = Field(
        default="output",
        description="Directory for output files"
    )
    
    # Analysis Settings
    top_n_recommendations: int = Field(
        default=50,
        description="Number of top recommendations to include"
    )
    min_roi_threshold: float = Field(
        default=20.0,
        description="Minimum ROI percentage threshold for recommendations"
    )
    min_automation_score: float = Field(
        default=0.3,
        description="Minimum automation potential score"
    )
    
    class Config:
        env_file = ".env"
        env_file_encoding = "utf-8"
        extra = "ignore"


# Cached settings instance
_settings: Optional[Settings] = None


def get_settings() -> Settings:
    """Get application settings (cached)"""
    global _settings
    if _settings is None:
        _settings = Settings()
    return _settings


def reload_settings() -> Settings:
    """Force reload settings from environment"""
    global _settings
    load_dotenv(env_path, override=True)
    _settings = Settings()
    return _settings


def print_config_status():
    """Print current configuration status"""
    settings = get_settings()
    print("=" * 50)
    print("Configuration Status")
    print("=" * 50)
    print(f"ServiceNow URL: {settings.servicenow_instance_url}")
    print(f"ServiceNow User: {settings.servicenow_username}")
    print(f"ServiceNow Pass: {'*' * len(settings.servicenow_password) if settings.servicenow_password else '[NOT SET]'}")
    print(f"OpenAI API Key: {'*' * 10 + settings.openai_api_key[-4:] if len(settings.openai_api_key) > 4 else '[NOT SET]'}")
    print(f"OpenAI Model: {settings.openai_model}")
    print(f"Use Mock Data: {settings.use_mock_data}")
    print("=" * 50)


if __name__ == "__main__":
    print_config_status()
