"""
LangGraph Agent for ServiceNow Application Portfolio Analysis
Orchestrates the analysis workflow and generates automation recommendations
"""

import os
import json
from typing import Annotated, TypedDict, Literal, Any
from datetime import datetime

from langchain_core.messages import HumanMessage, AIMessage, SystemMessage, ToolMessage
from langchain_openai import ChatOpenAI
from langgraph.graph import StateGraph, START, END
from langgraph.graph.message import add_messages
from langgraph.prebuilt import ToolNode

from config import get_settings
from tools import ALL_TOOLS, set_client
from servicenow_client import ServiceNowClient


# Agent State Definition
class AgentState(TypedDict):
    """State maintained throughout the agent's execution"""
    messages: Annotated[list, add_messages]
    portfolio_data: dict
    analysis_results: dict
    recommendations: list
    current_step: str
    error: str | None


# System prompt for the agent
SYSTEM_PROMPT = """You are an expert Enterprise Architect and Automation Strategist analyzing an application portfolio for a large financial services company. Your goal is to help CIOs and CFOs identify which applications should be automated using Generative AI and other automation technologies.

Your analysis should focus on:

1. **Business Value Assessment**
   - Identify applications with high manual effort that can benefit from automation
   - Calculate potential ROI and cost savings
   - Consider business criticality and risk factors

2. **Automation Opportunity Identification**
   - Document processing and data extraction
   - Report generation and distribution
   - Customer service and query handling
   - Compliance and audit workflows
   - Reconciliation and data matching
   - Approval workflows and notifications

3. **Prioritization Criteria**
   - High ROI potential (>50% recommended as priority)
   - Low implementation risk for critical systems
   - Quick wins with fast payback periods (<12 months)
   - Strategic alignment with digital transformation goals

4. **Risk Considerations**
   - Business criticality (be more cautious with Critical systems)
   - Regulatory requirements (SOX, GDPR, etc.)
   - Integration complexity
   - Change management requirements

When analyzing, use the available tools to:
1. First get the portfolio summary to understand the landscape
2. Identify high automation candidates
3. Find top ROI opportunities
4. Drill into specific categories or departments as needed
5. Build comprehensive recommendations

Always provide data-driven recommendations with specific metrics (ROI %, savings in USD, payback period).
"""


class ApplicationAnalyzerAgent:
    """LangGraph-based agent for application portfolio analysis"""
    
    def __init__(self, client: ServiceNowClient = None):
        settings = get_settings()
        
        # Initialize ServiceNow client
        self.client = client or ServiceNowClient()
        set_client(self.client)
        
        # Initialize LLM
        api_key = settings.openai_api_key or os.getenv("OPENAI_API_KEY")
        if not api_key:
            print("⚠️  Warning: OpenAI API key not set. Using mock analysis mode.")
            self.llm = None
        else:
            self.llm = ChatOpenAI(
                model=settings.openai_model,
                temperature=0,
                api_key=api_key
            )
            self.llm = self.llm.bind_tools(ALL_TOOLS)
        
        # Build the graph
        self.graph = self._build_graph()
        self.compiled_graph = self.graph.compile()
    
    def _build_graph(self) -> StateGraph:
        """Build the LangGraph workflow"""
        graph = StateGraph(AgentState)
        
        # Add nodes
        graph.add_node("analyze", self._analyze_node)
        graph.add_node("tools", ToolNode(ALL_TOOLS))
        graph.add_node("synthesize", self._synthesize_node)
        
        # Add edges
        graph.add_edge(START, "analyze")
        graph.add_conditional_edges(
            "analyze",
            self._should_continue,
            {
                "tools": "tools",
                "synthesize": "synthesize",
                "end": END
            }
        )
        graph.add_edge("tools", "analyze")
        graph.add_edge("synthesize", END)
        
        return graph
    
    def _analyze_node(self, state: AgentState) -> dict:
        """Main analysis node - invokes LLM with tools"""
        if self.llm is None:
            # Mock mode - return analysis without LLM
            return self._mock_analyze(state)
        
        messages = state["messages"]
        
        # Add system message if not present
        if not messages or not isinstance(messages[0], SystemMessage):
            messages = [SystemMessage(content=SYSTEM_PROMPT)] + messages
        
        response = self.llm.invoke(messages)
        return {"messages": [response], "current_step": "analyzing"}
    
    def _mock_analyze(self, state: AgentState) -> dict:
        """Mock analysis when no LLM is available"""
        from tools import (
            get_portfolio_summary, 
            get_high_automation_candidates,
            get_top_roi_opportunities,
            calculate_portfolio_automation_summary
        )
        
        # Get data using tools directly
        summary = get_portfolio_summary.invoke({})
        candidates = get_high_automation_candidates.invoke({"min_score": 0.6, "limit": 50})
        top_roi = get_top_roi_opportunities.invoke({"limit": 50})
        portfolio_analysis = calculate_portfolio_automation_summary.invoke({})
        
        analysis_text = f"""
## Application Portfolio Analysis Complete

### Executive Summary
I have analyzed **{summary['total_applications']}** applications in your enterprise portfolio.

### Key Findings

**Portfolio Overview:**
- Total Applications: {summary['total_applications']}
- Total Annual IT Spend: ${summary['total_annual_cost_usd']:,.2f}
- Total Users Supported: {summary['total_user_count']:,}

**Automation Opportunity:**
- Total Potential Annual Savings: ${portfolio_analysis['total_annual_savings_potential']:,.2f}
- Total Implementation Investment: ${portfolio_analysis['total_implementation_cost']:,.2f}
- Portfolio-Wide ROI: {portfolio_analysis['overall_portfolio_roi']:.1f}%

### Top 10 Automation Recommendations

| Rank | Application | Category | ROI % | Annual Savings | Payback |
|------|-------------|----------|-------|----------------|---------|
"""
        for i, app in enumerate(top_roi[:10], 1):
            analysis_text += f"| {i} | {app['name'][:30]} | {app['category']} | {app['roi_percentage']:.0f}% | ${app['annual_savings']:,} | {app['payback_months']:.1f}mo |\n"
        
        analysis_text += f"""

### Priority Tiers

**High Priority ({portfolio_analysis['priority_tiers']['high']['count']} apps)**
- Apps with automation score ≥0.7 and ROI ≥50%
- Total Savings: ${portfolio_analysis['priority_tiers']['high']['total_savings']:,}

**Medium Priority ({portfolio_analysis['priority_tiers']['medium']['count']} apps)**
- Apps with automation score 0.5-0.7 and ROI 20-50%
- Total Savings: ${portfolio_analysis['priority_tiers']['medium']['total_savings']:,}

### Category Analysis

"""
        for cat, data in portfolio_analysis['by_category'].items():
            analysis_text += f"- **{cat}**: {data['count']} apps, ${data['savings']:,} potential savings, {data['avg_roi']:.0f}% avg ROI\n"

        # Store data for report generation
        return {
            "messages": [AIMessage(content=analysis_text)],
            "portfolio_data": summary,
            "analysis_results": {
                "portfolio_summary": portfolio_analysis,
                "top_candidates": candidates,
                "top_roi": top_roi
            },
            "recommendations": top_roi[:50],
            "current_step": "complete"
        }
    
    def _should_continue(self, state: AgentState) -> Literal["tools", "synthesize", "end"]:
        """Determine next step based on state"""
        messages = state.get("messages", [])
        if not messages:
            return "end"
        
        last_message = messages[-1]
        
        # Check if we're in mock mode (analysis complete)
        if state.get("current_step") == "complete":
            return "end"
        
        # If LLM wants to use tools
        if hasattr(last_message, "tool_calls") and last_message.tool_calls:
            return "tools"
        
        # If we have enough analysis, synthesize
        if state.get("analysis_results"):
            return "synthesize"
        
        # Check message count - if we've had enough back and forth
        ai_messages = [m for m in messages if isinstance(m, AIMessage)]
        if len(ai_messages) >= 5:
            return "synthesize"
        
        return "end"
    
    def _synthesize_node(self, state: AgentState) -> dict:
        """Synthesize final recommendations"""
        if self.llm is None:
            return state
        
        synthesis_prompt = """Based on the analysis performed, provide a final executive summary with:
1. Top 10 applications recommended for Gen AI automation
2. Estimated total ROI and savings potential
3. Suggested implementation roadmap (Quick Wins, Strategic, Long-term)
4. Key risks and mitigation strategies

Format the response for a CIO/CFO audience - be concise and data-driven."""
        
        messages = state["messages"] + [HumanMessage(content=synthesis_prompt)]
        response = self.llm.invoke(messages)
        
        return {
            "messages": [response],
            "current_step": "synthesized"
        }
    
    def run(self, query: str = None) -> dict:
        """Run the analysis workflow"""
        if query is None:
            query = """Analyze our application portfolio and identify the top opportunities for Gen AI automation. 
Focus on applications with:
1. High automation potential
2. Strong ROI (>30%)
3. Reasonable payback period (<18 months)

Provide specific recommendations for CIO/CFO decision-making."""
        
        initial_state = {
            "messages": [HumanMessage(content=query)],
            "portfolio_data": {},
            "analysis_results": {},
            "recommendations": [],
            "current_step": "starting",
            "error": None
        }
        
        try:
            result = self.compiled_graph.invoke(initial_state)
            return result
        except Exception as e:
            return {
                **initial_state,
                "error": str(e),
                "current_step": "error"
            }
    
    def get_analysis_for_report(self) -> dict:
        """Run analysis and return structured data for report generation"""
        result = self.run()
        
        # Extract data from the state
        return {
            "portfolio_data": result.get("portfolio_data", {}),
            "analysis_results": result.get("analysis_results", {}),
            "recommendations": result.get("recommendations", []),
            "messages": [
                msg.content for msg in result.get("messages", [])
                if isinstance(msg, (AIMessage, HumanMessage))
            ],
            "timestamp": datetime.now().isoformat()
        }


def create_agent(client: ServiceNowClient = None) -> ApplicationAnalyzerAgent:
    """Factory function to create an agent"""
    return ApplicationAnalyzerAgent(client)


if __name__ == "__main__":
    from rich.console import Console
    from rich.markdown import Markdown
    
    console = Console()
    
    console.print("\n[bold cyan]🤖 ServiceNow AI Application Analyzer[/bold cyan]\n")
    console.print("[dim]Initializing agent...[/dim]\n")
    
    agent = create_agent()
    
    console.print("[yellow]Running portfolio analysis...[/yellow]\n")
    result = agent.run()
    
    # Display results
    for msg in result.get("messages", []):
        if isinstance(msg, AIMessage):
            console.print(Markdown(msg.content))
    
    if result.get("error"):
        console.print(f"\n[red]Error: {result['error']}[/red]")

